﻿namespace SensoryMAUI;

public partial class MainPage : ContentPage
{
    public MainPage()
    {
        InitializeComponent();

        pokażInformacjeOUrządzeniu();
        aktualizujInformacjeOStanieBaterii(
            null,
            new BatteryInfoChangedEventArgs(
                Battery.Default.ChargeLevel,
                Battery.Default.State,
                Battery.Default.PowerSource));
    }

    private void ContentPage_Loaded(object sender, EventArgs e)
    {
        DeviceDisplay.Current.MainDisplayInfoChanged += (object sender, DisplayInfoChangedEventArgs e) => { pokażInformacjeOWyświetlaczu(); };
        pokażInformacjeOWyświetlaczu(); //potrzebne w Windows        
    }

    private void pokażInformacjeOUrządzeniu()
    {        
        System.Text.StringBuilder sb = new System.Text.StringBuilder();
        sb.AppendLine("Informacje o urządzeniu:");
        sb.AppendLine($"Model: {DeviceInfo.Current.Model}");
        sb.AppendLine($"Producent: {DeviceInfo.Current.Manufacturer}");
        sb.AppendLine($"Nazwa: {DeviceInfo.Name}");
        sb.AppendLine($"Wersja systemu: {DeviceInfo.VersionString}");        
        sb.AppendLine($"Idiom: {DeviceInfo.Current.Idiom}");
        sb.AppendLine($"Platforma: {DeviceInfo.Current.Platform}");
        sb.AppendLine($"Typ urządzenia: {((DeviceInfo.Current.DeviceType == DeviceType.Virtual)? "wirtualne":"rzeczywiste")}");
        lbUrządzenie.Text = sb.ToString();
    }

    private void pokażInformacjeOWyświetlaczu()
    {
        System.Text.StringBuilder sb = new System.Text.StringBuilder();
        sb.AppendLine("Informacje o wyświetlaczu:");
        sb.AppendLine($"Rozdzielczość: {DeviceDisplay.Current.MainDisplayInfo.Width} x {DeviceDisplay.Current.MainDisplayInfo.Height} px");
        sb.AppendLine($"Gęstość: {DeviceDisplay.Current.MainDisplayInfo.Density}");
        sb.AppendLine($"Orientacja: {DeviceDisplay.Current.MainDisplayInfo.Orientation}");
        sb.AppendLine($"Obrót: {DeviceDisplay.Current.MainDisplayInfo.Rotation}");
        sb.AppendLine($"Częstość odświeżania: {DeviceDisplay.Current.MainDisplayInfo.RefreshRate} Hz");      

        lbEkran.Text = sb.ToString();
    }

    #region Bateria
    private bool czyZmianyStanuBateriiSąObserwowane;

    private void batterySwitch_Toggled(object sender, ToggledEventArgs e)
    {
        if (!czyZmianyStanuBateriiSąObserwowane) Battery.Default.BatteryInfoChanged += aktualizujInformacjeOStanieBaterii;
        else Battery.Default.BatteryInfoChanged -= aktualizujInformacjeOStanieBaterii;
        czyZmianyStanuBateriiSąObserwowane = !czyZmianyStanuBateriiSąObserwowane;
    }
    
    private void aktualizujInformacjeOStanieBaterii(object sender, BatteryInfoChangedEventArgs e)
    {
        lbBateriaStan.Text = e.State switch
        {
            BatteryState.Charging => "Ładowanie",
            BatteryState.Discharging => "Rozładowywanie (ładowarka nie jest podłączona)",
            BatteryState.Full => "Pełna",
            BatteryState.NotCharging => "Brak ładowania (łądowarka może być podłączona)",
            BatteryState.NotPresent => "Bateria nie jest obecna",
            BatteryState.Unknown => "Stan baterii nie jest znany",
            _ => "Nierozpoznany stan"
        };

        lbBateriaPoziom.Text = $"Bateria jest naładowana w {e.ChargeLevel * 100}%";
        pbBateriaPoziom.Progress = e.ChargeLevel;
    }
    #endregion

    #region Akcelerometr
    private void accelerometrSwitch_Toggled(object sender, ToggledEventArgs e)
    {
        if (Accelerometer.Default.IsSupported)
        {
            if (!Accelerometer.Default.IsMonitoring)
            {
                // Turn on accelerometer
                Accelerometer.Default.ReadingChanged += pokażOdczytPrzyspieszenia;
                Accelerometer.Default.ShakeDetected += sygnalizujPotrząśnięcie;
                Accelerometer.Default.Start(SensorSpeed.UI);
            }
            else
            {
                // Turn off accelerometer
                Accelerometer.Default.Stop();
                Accelerometer.Default.ReadingChanged -= pokażOdczytPrzyspieszenia;
                Accelerometer.Default.ShakeDetected -= sygnalizujPotrząśnięcie;
            }
        }
    }

    private void pokażOdczytPrzyspieszenia(object sender, AccelerometerChangedEventArgs e)
    {
        lbAkcelerometr.Text = $"Wersor przyspieszenia:\n\tX = {e.Reading.Acceleration.X}\n\tY = {e.Reading.Acceleration.Y}\n\tZ = {e.Reading.Acceleration.Z}\nDługość wektora: {9.81 * e.Reading.Acceleration.Length()}";
        pbPrzyspieszenie.Progress = e.Reading.Acceleration.Length() / 10;
        if(DateTime.Now > czasOstatniegoPotrząśnięcia.AddSeconds(3)) lbPotrząsanie.Text = $"---";
    }

    private DateTime czasOstatniegoPotrząśnięcia;

    private void sygnalizujPotrząśnięcie(object sender, EventArgs e)
    {
        lbPotrząsanie.Text = $"Wykryto potrząsanie urządzeniem";
        czasOstatniegoPotrząśnięcia = DateTime.Now;
        przełączStanLatarki();
        sygnalizujWibracjami();
    }
    #endregion

    #region Latarka i wibracje
    private bool latarkaWłączona = false;

    private async void przełączStanLatarki()
    {
        try
        {
            latarkaWłączona = !latarkaWłączona;
            if (latarkaWłączona) await Flashlight.Default.TurnOnAsync();
            else await Flashlight.Default.TurnOffAsync();            
        }
        catch (FeatureNotSupportedException)
        {
            await DisplayAlert(Title, "Urządzenie nie ma latarki", "OK");
        }
        catch (PermissionException)
        {
            await DisplayAlert(Title, "Aplikacja nie ma uprawnień do latarki", "OK");
        }
        catch (Exception)
        {
            await DisplayAlert(Title, "Włączenie lub wyłączenie latarki nie jest możliwe", "OK");
        }
    }

    private void flashlightButton_Clicked(object sender, EventArgs e) => przełączStanLatarki();

    private void sygnalizujWibracjami()
    {
        Vibration.Default.Vibrate(TimeSpan.FromSeconds(0.3));        
    }
    #endregion

    #region Barometr
    private void barometrSwitch_Toggled(object sender, ToggledEventArgs e)
    {
        if (Barometer.Default.IsSupported)
        {
            if (!Barometer.Default.IsMonitoring)
            {
                Barometer.Default.ReadingChanged += Barometer_ReadingChanged;
                Barometer.Default.Start(SensorSpeed.UI);
            }
            else
            {
                Barometer.Default.Stop();
                Barometer.Default.ReadingChanged -= Barometer_ReadingChanged;
            }
        }
        else lbBarometr.Text = "Urządzenie nie posiada barometru";
    }

    private void Barometer_ReadingChanged(object sender, BarometerChangedEventArgs e)
    {
        lbBarometr.Text = $"Barometr: {e.Reading}";
        pbCiśnienie.Progress = e.Reading.PressureInHectopascals / 2000;
    }
    #endregion

    #region Kompas
    private void compassSwitch_Toggled(object sender, ToggledEventArgs e)
    {
        if (Compass.Default.IsSupported)
        {
            if (!Compass.Default.IsMonitoring)
            {
                // Turn on compass
                Compass.Default.ReadingChanged += Compass_ReadingChanged;
                Compass.Default.Start(SensorSpeed.UI);
            }
            else
            {
                // Turn off compass
                Compass.Default.Stop();
                Compass.Default.ReadingChanged -= Compass_ReadingChanged;
            }
        }
        else lbKompas.Text = "Urządzenie nie posiada kompasu";
    }

    private void Compass_ReadingChanged(object sender, CompassChangedEventArgs e)
    {
        lbKompas.Text = $"Compass: {e.Reading.HeadingMagneticNorth}";
    }
    #endregion

    #region Orientacja
    private void orientacjaSwitch_Toggled(object sender, ToggledEventArgs e)
    {
        if (OrientationSensor.Default.IsSupported)
        {
            if (!OrientationSensor.Default.IsMonitoring)
            {
                OrientationSensor.Default.ReadingChanged += Orientation_ReadingChanged;
                OrientationSensor.Default.Start(SensorSpeed.UI);
            }
            else
            {
                OrientationSensor.Default.Stop();
                OrientationSensor.Default.ReadingChanged -= Orientation_ReadingChanged;
            }
        }
        else lbOrientacja.Text = "Urządzenie nie posiada czujnika magnetycznego";
    }

    private void Orientation_ReadingChanged(object sender, OrientationSensorChangedEventArgs e)
    {
        lbOrientacja.Text = $"Orientacja:\n\tX = {e.Reading.Orientation.X}\n\tY = {e.Reading.Orientation.Y}\n\tZ = {e.Reading.Orientation.Z}\n\tW = {e.Reading.Orientation.W}";        
    }
    #endregion

    #region Geolokalizacja
    private string opisPołożenia(Location położenie)
    {
        double odległośćOdTorunia = położenie.CalculateDistance(new Location(53.0163331, 18.5911742), DistanceUnits.Kilometers);
        return $"Współrzędne geograficzne\n\tSzerokość: {położenie.Latitude}\n\tDługość: {położenie.Longitude}\n\tWysokość: {położenie.Altitude}\n\tDokładność: {położenie.Accuracy}\n\tAzymut: {położenie.Course}\n\tOdległość od Torunia: {odległośćOdTorunia}";
    }

    private bool czyPołożenieJestAktualnieOdczytywane = false;
    private CancellationTokenSource tokenOdczytuPołożenia;

    public async Task<Location> PobierzPołożenie(bool użyjZapamiętanegoPołożenia = false)
    {
        try
        {
            czyPołożenieJestAktualnieOdczytywane = true;
            Location położenie;
            if (!użyjZapamiętanegoPołożenia)
            {
                GeolocationRequest żądanie = new GeolocationRequest(GeolocationAccuracy.Medium, TimeSpan.FromSeconds(10));
                tokenOdczytuPołożenia = new CancellationTokenSource();
                położenie = await Geolocation.Default.GetLocationAsync(żądanie, tokenOdczytuPołożenia.Token);
            }
            else
            {
                położenie = await Geolocation.Default.GetLastKnownLocationAsync();
            }
            if (położenie == null) throw new Exception("Niemożliwe odczytanie lokalizacji");
            else return położenie;
        }
        catch (FeatureNotSupportedException ex)
        {
            throw new Exception("Nie można odczytać lokalizacji na tym urządzeniu", ex);
        }
        catch (FeatureNotEnabledException ex)
        {
            throw new Exception("Odczyt lokalizacji nie jest dostępny", ex);
        }
        catch (PermissionException ex)
        {
            throw new Exception("Aplikacja nie ma uprawnień do odczytu lokalizacji", ex);
        }
        catch (Exception ex)
        {
            throw new Exception("Błąd w odczycie lokalizacji", ex);
        }
        finally
        {
            czyPołożenieJestAktualnieOdczytywane = false;
        }
    }

    public void AnulujŻądaniePobraniaLokalizacji()
    {
        if (czyPołożenieJestAktualnieOdczytywane && tokenOdczytuPołożenia != null && tokenOdczytuPołożenia.IsCancellationRequested == false)
            tokenOdczytuPołożenia.Cancel();
    }

    private async void pokażPołożenie()
    {
        btnOdświeżPołożenie.IsEnabled = false;
        Location lokalizacja = await PobierzPołożenie();
        lbPołożenie.Text = opisPołożenia(lokalizacja);
        btnOdświeżPołożenie.IsEnabled = true;
        btnOtwórzMapę.IsEnabled = true;
    }

    private void locationButton_Clicked(object sender, EventArgs e)
    {
        pokażPołożenie();
    }

    private async void mapButton_Clicked(object sender, EventArgs e)
    {
        btnOtwórzMapę.IsEnabled = false;
        Location położenie = await PobierzPołożenie();
        await położenie.OpenMapsAsync();
        btnOtwórzMapę.IsEnabled = true;
    }
    #endregion
}

