//*******************************************************
//*       Shortcut Component for Delphi 2.0             *
//*       by Elliott Shevin, Oak Park, Mich. USA        *
//*       April, 1999                                   *
//*       email: shevine@aol.com                        *
//*                                                     *
//*                                                     *
//*    This component incorporates the shortcut read    *
//*    function of TShellLink by Radek Voltr with       *
//*    shortcut creation code from Jordan Russell,      *
//*    who merits special thanks for reviewing and      *
//*    improving the code.                              *
//*                                                     *
//*    This is a freeware component. Use it any way     *
//*    you like, but please report errors and improve-  *
//*    ments to me, and acknowledge Radek and Jordan.   *
//*******************************************************

unit ShortcutLink;

{$IFNDEF VER80} {$IFNDEF VER90} {$IFNDEF VER93}
  {$DEFINE Delphi3orHigher}
{$ENDIF} {$ENDIF} {$ENDIF}

interface

uses
  Windows,
  Messages,
  SysUtils,
  Classes,
  Forms,
  {$IFNDEF Delphi3orHigher}
     OLE2,
  {$ELSE}
     ActiveX, ComObj,
  {$ENDIF}
  ShellAPI,
  ShlObj,
  CommCtrl,
  StdCtrls;

const
    SLR_NO_UI		= $0001;
    SLR_ANY_MATCH	= $0002;
    SLR_UPDATE          = $0004;

    SLGP_SHORTPATH	= $0001;
    SLGP_UNCPRIORITY	= $0002;
    Error_Message       = 'Unable to create .lnk file';

    //JR - added the following
    {$IFDEF Delphi3orHigher}
    IID_IPersistFile: TGUID = (
      D1:$0000010B; D2:$0000; D3:$0000; D4:($C0,$00,$00,$00,$00,$00,$00,$46));
    {$ENDIF}

type
EShortcutError = class(Exception);
TShowCmd = (scShowMaximized, scShowMinimized, scShowNormal);


type
  TShortcutLink = class(TComponent)
  private
    { Private declarations }
  protected
    { Protected declarations }
    fShortcutFile,
    fTarget,
    fWorkingDir,
    fDescription,
    fArguments,
    fIconLocation:String;
    fIconNumber,
    fShowCmd,
    fHotKey:Word;
    procedure fSetHotKey(c : string);
    function  fGetHotKey : string;
    procedure fSetShowCmd(c : TShowCmd);
    function  fGetShowCmd : TShowCmd;
    function  fGetDesktopFolder   : string;
    function  fGetProgramsFolder  : string;
    function  fGetStartFolder     : string;
    function  fGetStartupFolder   : string;
    function  fGetSpecialFolder(nFolder : integer) : string;
  public
    { Public declarations }
    procedure Read;
    procedure Write;
    property DesktopFolder   : string read fGetDesktopFolder;
    property ProgramsFolder  : string read fGetProgramsFolder;
    property StartFolder     : string read fGetStartFolder;
    property StartupFolder   : string read fGetStartupFolder;
  published
    { Published declarations }
    property ShortcutFile:String read fShortcutFile write fShortcutFile;
    property Target:String read fTarget write fTarget;
    property WorkingDir:String read fWorkingDir write fWorkingDir;
    property Description:String read fDescription write fDescription;
    property Arguments:String read fArguments write fArguments;
    property IconLocation:String read fIconLocation write fIconLocation;
    property HotKey:string read fGetHotKey write fSetHotKey;
    property ShowCmd:TShowCmd read fGetShowCmd write fSetShowCmd default scShowNormal;
    property IconNumber:word read fIconNumber write fIconNumber;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Win95', [TShortcutLink]);
end;


// This is the Read method, which reads the link to which
// fShortcutFile points. It was SetSelfPath in Radek Voltr's TShellLink
// component, where setting the ShortcutFile property caused the shortcut
// file to be read immediately.
procedure TShortcutLink.Read;
var
    X3:PChar;
    hresx:HResult;
    Psl:IShellLink;  //JR - changed from IShortcutLink to IShellLink
    Ppf:IPersistFile;
    Saver:Array [0..Max_Path] of WideChar;
    X1:Array [0..MAX_PATH-1] Of Char;  //JR - changed from 255 to MAX_PATH-1 (255 isn't big enough)
    Data:TWin32FindData;I,Y:INteger;W:Word;
begin
     //JR - changed CLSID_ShortcutLink and IID_IShortcutLink to CLSID_ShellLink and IID_IShellLinkA/IID_IShellLink
     hresx:=CoCreateInstance(CLSID_ShellLink,nil,CLSCTX_INPROC_SERVER,{$IFDEF Delphi3orHigher}IID_IShellLinkA{$ELSE}IID_IShellLink{$ENDIF},psl);
     If hresx<>0 then Exit;
     hresx:=psl.QueryInterface(IID_IPersistFile,ppf);
     If hresx<>0 then Exit;
     X3:=StrAlloc(MAX_PATH);  //JR - changed from 255 to MAX_PATH

     StrPCopy(X3,fShortcutFile);  // Changed from S in Radek Voltr's TShellLink component
     MultiByteToWideChar(CP_ACP,0,X3,-1,Saver,Max_Path);
     hresx:=ppf.Load(Saver,STGM_READ);
     If hresx<>0 then
        begin
             raise EShortcutError.Create('Unable to open link file');
             Exit;
     end;
     hresx:=psl.Resolve(0,SLR_ANY_MATCH);
     If hresx<>0 then Exit;
     hresx:= psl.GetWorkingDirectory(@X1,MAX_PATH );
     If hresx<>0 then
        begin
             raise EShortcutError.Create('Error in getting WorkingDir');
             Exit;
        end;
     fWorkingDir:=StrPas(@X1);

     hresx:= psl.GetPath( @X1,MAX_PATH,Data,SLGP_UNCPRIORITY);
     If hresx<>0 then
        begin
             raise EShortcutError.Create('Error in getting Target');
             Exit;
        end;
        fTarget:=StrPas(@X1);

     hresx:=psl.GetIconLocation(@X1,MAX_PATH,I);  //JR - changed @I to I
     If hresx<>0 then
        begin
             raise EShortcutError.Create('Error in getting icon data');
             Exit;
     end;
     fIconLocation:=StrPas(@X1);
     fIconNumber:=I;

     hresx:= psl.GetDescription(@X1,MAX_PATH );
     If hresx<>0 then
        begin
             raise EShortcutError.Create('Error in get Description');
             Exit;
     end;
     fDescription:=StrPas(@X1);

     Y:=0;
     hresx:= psl.GetShowCmd(Y);  //JR - changed @Y to Y
     If hresx<>0 then
        begin
             raise EShortcutError.Create('Error in getting ShowCmd');
             Exit;
        end;
     fShowCmd:=Y;

     W:=0;
     hresx:= psl.GetHotKey(W);  //JR - changed @W to W
     If hresx<>0 then
        begin
             raise EShortcutError.Create('Error in geting HotKey');
             Exit;
        end;
     fHotKey:=W;
     if fHotKey = 0
        then HotKey := ' '
        else HotKey := chr(fHotKey);

     hresx:= psl.GetArguments(@X1,MAX_PATH );
     If hresx<>0 then
        begin
             raise EShortcutError.Create('Error in getting Arguments');
             Exit;
        end;

     fArguments:=StrPas(@X1);

//JR - added 'ifndef' around this
{$IFNDEF Delphi3orHigher}
     ppf.release;
     psl.release;
{$ENDIF}
     StrDispose(X3);
end;


// The Write method is adapted from code in Jordan Russell's
// Inno Setup.
procedure TShortcutLink.Write;
var
  aISL: IShellLink;
  aIPF: IPersistFile;
{$IFNDEF Delphi3OrHigher}
  aPidl: PItemIDList;
  WideFilename: array[0..MAX_PATH-1] of WideChar;
{$ELSE}
  Obj: IUnknown;
  WideFilename: WideString;
{$ENDIF}

begin
    // Get an IShellLink interface to make the shortcut.
    // The methods differ between Delphi 2 and later releases.
{$IFNDEF Delphi3OrHigher}
    if not SUCCEEDED(CoCreateInstance(CLSID_ShellLink, nil,
              CLSCTX_INPROC_SERVER, IID_IShellLink, aISL))
         then raise EShortcutError.Create (Error_Message);
{$ELSE}
    Obj := CreateComObject(CLSID_ShellLink);
    aiSL := Obj as IShellLink;
{$ENDIF}

    //JR - added 'try'
    try

      // Now we have an IShellLink interface, so we can set it up as we like.
      // Set the target.
      aISL.SetPath(Pchar(fTarget));
      // Set the working directory ("Start in")
      aISL.SetWorkingDirectory(PChar(fWorkingDir));
      // Set the command-line params
      aISL.SetArguments(Pchar(fArguments));
      // Set the description
      aISL.SetDescription(Pchar(fDescription));
      // Set the show command
      aISL.SetShowCmd(fShowCmd);
      // Set the hot-key
      aISL.SetHotKey(((HOTKEYF_ALT or HOTKEYF_CONTROL) shl 8) or fHotKey);
      // Set the icon location
      aISL.SetIconLocation(Pchar(fIconLocation), fIconNumber);

      // The shortcut IShellLink is now all set up.
      // We get an IPersistFile interface from it, and use it to save the link.
      // Delphi 2 differs from later releases.
{$IFNDEF Delphi3OrHigher}
      if aISL.QueryInterface(IID_IPersistFile, aIPF) <> S_OK
          then raise EShortcutError.Create (Error_Message)
          else MultiByteToWideChar (CP_ACP, 0, PChar(fShortcutFile), -1,
                       WideFilename, MAX_PATH);
{$ELSE}
      aiPF := Obj as IPersistFile;
      WideFilename := fShortcutFile;
{$ENDIF}

      try
{$IFNDEF Delphi3OrHigher}
        if aIPF.Save (WideFilename, True) <> S_OK
{$ELSE}
        if aIPF.Save (PWideChar(WideFilename), True) <> S_OK
{$ENDIF}
            then raise EShortcutError.Create (Error_Message);
      finally
{$IFNDEF Delphi3OrHigher}
        aIPF.Release;  // Only needed for D2--later releases do this implicitly.
{$ENDIF}
      end;

      //JR - added this:
    finally
{$IFNDEF Delphi3OrHigher}
      aISL.Release;  // Only needed for D2--later releases do this implicitly.
{$ENDIF}
    end;
end;    

function  TShortcutLink.fGetDesktopFolder   : string;
begin
    result := fGetSpecialFolder(CSIDL_DESKTOPDIRECTORY);
end;

function  TShortcutLink.fGetProgramsFolder  : string;
begin
    result := fGetSpecialFolder(CSIDL_PROGRAMS);
end;

function  TShortcutLink.fGetStartFolder     : string;
begin
    result := fGetSpecialFolder(CSIDL_STARTMENU);
end;

function  TShortcutLink.fGetStartupFolder   : string;
begin
    result := fGetSpecialFolder(CSIDL_STARTUP);
end;

function TShortcutLink.fGetSpecialFolder(nFolder : integer) : string;
var
  aPidl: PItemIDList;
  handle : THandle;
  TC     : TComponent;
  fLinkDir : string;
begin
  // Get the folder location (as a PItemIDList)
  TC := self.owner;
  handle := (TC as TForm).handle;
  if SUCCEEDED(SHGetSpecialFolderLocation(handle, nFolder, aPidl))
     then begin
    // Get the actual path of the desktop directory from the PItemIDList
       SetLength(fLinkDir, MAX_PATH); // SHGetPathFromIDList assumes MAX_PATH buffer
       SHGetPathFromIDList(aPidl, PChar(fLinkDir)); // Do it
       SetLength(fLinkDir, StrLen(PChar(fLinkDir)));
       result := fLinkDir;
  end;
end;

procedure TShortcutLink.fSetHotKey(c : string);
var
   s : string[1];
   c2 : char;
begin
     s := c;
     s := uppercase(s);
     c2 := s[1];
     fHotKey := ord(c2);
end;

function  TShortcutLink.fGetHotKey : string;
begin
     if fHotKey = 0 then fHotKey := ord(' ');
     result := chr(fHotKey);
end;

procedure TShortcutLink.fSetShowCmd(c : TShowCmd);
begin
     case c of
          scSHOWMAXIMIZED     : fShowCmd := SW_ShowMaximized;
          scSHOWMINIMIZED     : fShowCmd := SW_ShowMinimized;
          scSHOWNORMAL        : fShowCmd := SW_ShowNormal;
     end;
end;

function  TShortcutLink.fGetShowCmd : TShowCmd;
begin
     case fShowCmd of
          SW_SHOWMAXIMIZED     : result := scShowMaximized;
          SW_SHOWMINIMIZED     : result := scShowMinimized;
          SW_SHOWNORMAL        : result := scShowNormal;
     //JR - added the following two lines to get rid of the compiler warning
     else
       result := scShowNormal;
     end;
end;

initialization
  CoInitialize(nil); // Must initialize COM or CoCreateInstance won't work
finalization
  CoUninitialize; // Symmetric uninitialize

end.
