﻿//Jacek Matulewski (2009), e-mail: jacek@fizyka.umk.pl

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;

//zrownoleglenie TPL (Parallel.For)
using System.Threading.Tasks;
using System.Threading;

namespace ITA106
{
    public enum Algorytm { algorytmEulera, algorytmVerleta };
    public enum SposobZrownoleglania { Sekwencyjne, ParallelFor, Zadania, FabrykaZadan, Watki, PulaWatkow };

    /********************************************************/
    /*                  Punkt materialny                    */
    /********************************************************/

    public struct PunktMaterialny
    {
        #region Pola
        private Vector3 polozenie;
        private Vector3 predkosc;
        private Vector3 poprzedniePolozenie;
        private Vector3 nastepnePolozenie;
        private Vector3 nastepnaPredkosc;
        private float masa;
        private float promien;
        private Color kolor;
        #endregion

        #region Konstruktor
        public PunktMaterialny(Vector3 polozenie, Vector3 predkosc, float masa, float promien, Color kolor)
        {
            this.numerKroku = 0;
            this.polozenie = polozenie;
            this.poprzedniePolozenie = polozenie;
            this.predkosc = predkosc;
            this.masa = masa;
            this.promien = promien;
            this.kolor = kolor;

            this.nastepnePolozenie = Vector3.Zero;
            this.nastepnaPredkosc = Vector3.Zero;
        }
        static public PunktMaterialny Domyslny = new PunktMaterialny(Vector3.Zero, Vector3.Zero, 1, 0, Color.White);
        #endregion

        #region Wlasnosci
        public Vector3 Polozenie
        {
            get
            {
                return polozenie;
            }
            set
            {
                polozenie = value;
            }
        }
        public Vector3 PoprzedniePolozenie
        {
            get
            {
                return poprzedniePolozenie;
            }
            set
            {
                poprzedniePolozenie = value;
            }
        }
        public Vector3 NastepnePolozenie
        {
            get
            {
                return nastepnePolozenie;
            }
            set
            {
                nastepnePolozenie = value;
            }
        }
        public Vector3 Predkosc
        {
            get
            {
                return predkosc;
            }
            set
            {
                predkosc = value;
            }
        }
        public float Masa
        {
            get
            {
                return masa;
            }
            set
            {
                masa = value;
            }
        }
        public float Promien
        {
            get
            {
                return promien;
            }
            set
            {
                promien = value;
            }
        }
        public Color Kolor
        {
            get
            {
                return kolor;
            }
            set
            {
                kolor = value;
            }
        }
        #endregion

        #region Dynamika
        float sqr(float x)
        {
            return x * x;
        }

        private int numerKroku;
        private void PrzygotujRuch_Euler(Vector3 przyspieszenie, float krokCzasowy)
        {
            nastepnaPredkosc = predkosc + przyspieszenie * krokCzasowy;
            nastepnePolozenie = polozenie + nastepnaPredkosc * krokCzasowy;
        }
        private void PrzygotujRuch_Verlet(Vector3 przyspieszenie, float krokCzasowy)
        {
            if (numerKroku == 0) PrzygotujRuch_Euler(przyspieszenie, krokCzasowy);
            else
            {
                nastepnePolozenie = 2 * polozenie - poprzedniePolozenie + przyspieszenie * sqr(krokCzasowy);
                nastepnaPredkosc = (nastepnePolozenie - poprzedniePolozenie) / (2 * krokCzasowy);
            }
        }
        public void PrzygotujRuch(Vector3 sila, float krokCzasowy, Algorytm algorytm)
        {
            Vector3 przyspieszenie = sila / masa;
            switch (algorytm)
            {
                case Algorytm.algorytmEulera:
                    PrzygotujRuch_Euler(przyspieszenie, krokCzasowy);
                    break;
                case Algorytm.algorytmVerleta:
                    PrzygotujRuch_Verlet(przyspieszenie, krokCzasowy);
                    break;
            }
        }
        public void WykonajRuch()
        {
            poprzedniePolozenie = polozenie;
            polozenie = nastepnePolozenie;
            predkosc = nastepnaPredkosc;
            ++numerKroku;
        }
        public int NumerKroku
        {
            get
            {
                return numerKroku;
            }
        }
        #endregion
    }

    /********************************************************/
    /*            Zbior punktow materialnych                */
    /********************************************************/

    public abstract class ZbiorPunktowMaterialnych
    {
        #region Pola
        protected int ilosc;
        protected PunktMaterialny[] punkty;
        protected bool[] wiezy;        
        private SposobZrownoleglania obliczeniaRownolegle;
        #endregion

        #region Metody wirtualne (do nadpisania)
        protected abstract Vector3 Sila(int indeks);
        protected virtual void PrzedKrokiemNaprzod(float krokCzasowy) { }
        protected virtual void PoPrzygotowaniuRuchu(float krokCzasowy) { }
        protected virtual void PoKrokuNaprzod(float krokCzasowy) { }
        #endregion

        #region Konstruktor
        public ZbiorPunktowMaterialnych(int ilosc, SposobZrownoleglania obliczeniaRownolegle = SposobZrownoleglania.Sekwencyjne)
        {
            this.ilosc = ilosc;
            this.punkty = new PunktMaterialny[ilosc];
            this.wiezy = new bool[ilosc];
            for (int i = 0; i < ilosc; i++)
            {
                wiezy[i] = false;
                punkty[i] = PunktMaterialny.Domyslny;
            }
            this.obliczeniaRownolegle = obliczeniaRownolegle;
        }
        #endregion

        #region Dynamika
        public virtual void PrzygotujRuch(float krokCzasowy, Algorytm algorytm)
        {
            int iloscProcesorow = System.Environment.ProcessorCount;

            #region Akcja wykonywana przez watki i zadania
            Action<object> a = (object o) =>
                {
                    //if (!(o is int)) throw new ArgumentException("Argument o ({0}) powinien być liczbą całkowitą", o.ToString());
                    int n = (int)o;

                    int d = ilosc / iloscProcesorow;
                    //Console.WriteLine("[{0},{1})", n * d, (n + 1) * d);

                    for (int i = n * d; i < (n + 1) * d; ++i)
                        if (!wiezy[i])
                        {
                            //Console.WriteLine("krok=" + punkty[0].NumerKroku.ToString() +", i=" + i.ToString());
                            punkty[i].PrzygotujRuch(Sila(i), krokCzasowy, algorytm);
                        }
                    //Console.WriteLine("Zadanie nr {0} skonczone.", Task.CurrentId);
                };
            #endregion

            //tu wybór podejscia do zrownoleglania; jednak wszystkie sa wolniejsze od sekwencyjnego
            switch(obliczeniaRownolegle)
            {
                case SposobZrownoleglania.Sekwencyjne:
                    for (int i = 0; i < ilosc; ++i) if (!wiezy[i]) punkty[i].PrzygotujRuch(Sila(i), krokCzasowy, algorytm);
                    break;
                    
                case SposobZrownoleglania.ParallelFor:
                    //Parallel.For
                    ParallelOptions po = new ParallelOptions();
                    po.MaxDegreeOfParallelism = iloscProcesorow;
                    Parallel.For(0, ilosc, po, (i) => { if (!wiezy[i]) punkty[i].PrzygotujRuch(Sila(i), krokCzasowy, algorytm); });
                    break;

                case SposobZrownoleglania.Zadania:
                    //Zadania
                    {
                        Task[] zadania = new Task[iloscProcesorow];
                        for (int i = 0; i < iloscProcesorow; ++i)
                        {
                            zadania[i] = new Task(a, i);
                            zadania[i].Start();
                        }
                        Task.WaitAll(zadania);
                        break;
                    }

                case SposobZrownoleglania.FabrykaZadan:
                    //Fabryka zadan
                    {
                        Task[] zadania = new Task[iloscProcesorow];
                        for (int i = 0; i < iloscProcesorow; ++i)
                        {
                            zadania[i] = Task.Factory.StartNew(a, i);
                        }
                        Task.WaitAll(zadania);
                        break;
                    }
                case SposobZrownoleglania.Watki:
                    //Watki
                    Thread[] watki=new Thread[iloscProcesorow];
                    for (int i = 0; i < iloscProcesorow; ++i)
                    {
                        ParameterizedThreadStart ts=new ParameterizedThreadStart(a);
                        watki[i] = new Thread(ts); //lokalne tworzenie watkow - b. wolne
                        watki[i].Start(i);
                    }
                    foreach (Thread t in watki) t.Join();
                    break;

                case SposobZrownoleglania.PulaWatkow:
                    //Pula watkow
                    int ileDostepnychWatkowPuli = 0;
                    int ileWszystkichWatkowPuli = 0;
                    int ileDzialajacychWatkowPuli = 0;
                    int tmp = 0;
                    WaitCallback metodaWatku = new WaitCallback(a);
                    for (int i = 0; i < iloscProcesorow; ++i)
                        ThreadPool.QueueUserWorkItem(metodaWatku, i);
                    do
                    {
                        ThreadPool.GetAvailableThreads(out ileDostepnychWatkowPuli, out tmp);
                        ThreadPool.GetMaxThreads(out ileWszystkichWatkowPuli, out tmp);
                        ileDzialajacychWatkowPuli = ileWszystkichWatkowPuli - ileDostepnychWatkowPuli;
                    }
                    while (ileDzialajacychWatkowPuli > 0);
                    break;
            }
        }
        public void WykonajRuch()
        {
            /*if (obliczeniaRownolegle)
                Parallel.For(0, ilosc, (i) => { if (!wiezy[i]) punkty[i].WykonajRuch(); });
            else*/
                for (int i = 0; i < ilosc; ++i) if (!wiezy[i]) punkty[i].WykonajRuch();
        }
        public void KrokNaprzod(float krokCzasowy, Algorytm algorytm)
        {
            PrzedKrokiemNaprzod(krokCzasowy);
            PrzygotujRuch(krokCzasowy, algorytm);
            PoPrzygotowaniuRuchu(krokCzasowy);
            WykonajRuch();
            PoKrokuNaprzod(krokCzasowy);
        }
        #endregion

        #region Wlasnosci
        public int LiczbaPunktow
        {
            get
            {
                return ilosc;
            }
        }
        public PunktMaterialny this[int indeks]
        {
            get
            {
                return punkty[indeks];
            }
        }
        public Vector3 SrodekMasy
        {
            get
            {
                Vector3 srodekMasy = Vector3.Zero;
                for (int indeks = 0; indeks < ilosc; indeks++)
                    srodekMasy += punkty[indeks].Polozenie;
                srodekMasy /= (float)ilosc;
                return srodekMasy;
            }
        }
        public SposobZrownoleglania ObliczeniaRownolegle
        {
            get
            {
                return obliczeniaRownolegle;
            }
            set
            {
                obliczeniaRownolegle = value;
            }
        }
        #endregion

        #region Metody
        public void UstawWiezy(int indeks, bool ustalonaPozycja)
        {
            this.wiezy[indeks] = ustalonaPozycja;
        }

        protected void ZerujPredkoscSrednia()
        {
            int iloscZWiezami = 0;
            Vector3 predkoscSrednia = Vector3.Zero;
            for (int i = 0; i < ilosc; ++i)
            {
                if (!wiezy[i])
                    predkoscSrednia += punkty[i].Predkosc;
                else
                    iloscZWiezami++;
            }
            predkoscSrednia /= ilosc - iloscZWiezami;
            for (int i = 0; i < ilosc; ++i)
                if (!wiezy[i])
                    punkty[i].Predkosc -= predkoscSrednia;
        }
        #endregion

        #region Konwersja do tablicy werteksow
        public VertexPositionColor[] ToArrayOfVertices()
        {
            VertexPositionColor[] werteksy = new VertexPositionColor[ilosc];
            for (int i = 0; i < ilosc; ++i)
            {
                werteksy[i].Color = punkty[i].Kolor;
                werteksy[i].Position = punkty[i].Polozenie;
            }
            return werteksy;
        }
        #endregion
    }
}
