﻿using Microsoft.EntityFrameworkCore;
using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace LinqToSqlite
{
    class Osoba : ICloneable //DTO, POCO
    {
        [Key] public int Id; //primary key
        public string Imię, Nazwisko;
        public int NumerTelefonu;
        public int Wiek;

        public override string ToString()
        {
            return $"{Imię} {Nazwisko} ({Wiek}), nr telefonu: {NumerTelefonu}";
        }

        public object Clone()
        {
            return new Osoba() { Imię = this.Imię, Nazwisko = this.Nazwisko, Wiek = this.Wiek, NumerTelefonu = this.NumerTelefonu };
        }

        public static readonly Osoba OsobaPusta = new Osoba() { Id = -1, Imię = "", Nazwisko = "", NumerTelefonu = -1, Wiek = -1 };

        public override bool Equals(object? obj)
        {
            if (!(obj is Osoba)) return false;
            Osoba innaOsoba = (Osoba)obj;
            return
                innaOsoba.Id == Id &&
                innaOsoba.Imię.Equals(Imię) &&
                innaOsoba.Nazwisko.Equals(Nazwisko);
        }

        public override int GetHashCode()
        {
            return Id.GetHashCode() ^ Imię.GetHashCode() ^ Nazwisko.GetHashCode();
        }
    }

    class Adres
    {
        [Key] public int Id;
        public string Miasto, Ulica;
        public int NumerDomu;
        public int? NumerMieszkania;

        public override string ToString()
        {
            return $"{Miasto}, {Ulica} {NumerDomu}" + ((NumerMieszkania.HasValue) ? $"/{NumerMieszkania}" : "");
        }

        public override bool Equals(object? obj)
        {
            if (!(obj is Adres)) return false;
            Adres innyAdres = (Adres)obj;
            return
                innyAdres.Miasto.Equals(Miasto) &&
                innyAdres.Ulica.Equals(Ulica) &&
                innyAdres.NumerDomu.Equals(NumerDomu) &&
                innyAdres.NumerMieszkania.Equals(NumerMieszkania);
        }

        public override int GetHashCode()
        {
            return Miasto.GetHashCode() ^ Ulica.GetHashCode() ^ NumerDomu.GetHashCode() ^ NumerMieszkania.GetHashCode();
        }
    }

    static class Rozszerzenia
    {
        public static void Wyświetl<T>(this IEnumerable<T> list)
        {
            Console.WriteLine("Lista obiektów: ");
            foreach (T element in list)
            {
                if (element != null) Console.WriteLine(element.ToString());
                else Console.WriteLine("---");
            }
            Console.WriteLine();
        }
    }

    class BazaDanychOsóbDbContext : DbContext
    {
        public DbSet<Osoba> Osoby { get; set; }
        public DbSet<Adres> Adresy { get; set; }

        protected override void OnConfiguring(DbContextOptionsBuilder optionsBuilder)
        {
            //base.OnConfiguring(optionsBuilder);
            optionsBuilder.UseSqlite("Data Source=osoby.db");
        }
    }

    class BazaDanychOsób : IDisposable
    {
        private BazaDanychOsóbDbContext dbc = new BazaDanychOsóbDbContext();

        public BazaDanychOsób()
        {
            dbc.Database.EnsureCreated();
        }

        public void Dispose()
        {
            dbc.Dispose();
        }

#if DEBUG
        public Osoba[] Osoby { get => dbc.Osoby.ToArray(); }
        public Adres[] Adresy { get => dbc.Adresy.ToArray(); }
#endif

        #region CRUD = Create, Read+, Update, Delete
        public Osoba PobierzOsobę(int idOsoby)
        {
            return dbc.Osoby.FirstOrDefault(o => o.Id == idOsoby);
        }

        //indekser
        public Osoba this[int idOsoby]
        {
            get => PobierzOsobę(idOsoby);
        }

        public int[] IdentyfikatoryOsób
        {
            get => dbc.Osoby/*.OrderBy(o => o.Id)*/.Select(o => o.Id).OrderBy(id => id).ToArray();
        }

        public int DodajOsobę(Osoba osoba) //identyfikator w argumencie będzie ignorowany
        {
            if(osoba == null)
                throw new ArgumentNullException(nameof(osoba));
        }
        #endregion
    }
}
