﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Turing.Model
{
    public class Maszyna
    {
        public List<string> Ostrzeżenia { get; private set; } = new List<string>();

        private void dodajOstrzeżenie(string ostrzeżenie)
        {
            Ostrzeżenia.Add(ostrzeżenie);
            onDodaneOstrzeżenie();
        }

        public void WczytajProgram(string ścieżkaPliku)
        {
            program = Program.WczytajProgram(ścieżkaPliku);
            onStateChanged(null);
        }        

        #region Stan
        private List<char> taśma = new List<char>() { 'A', 'A', 'A', 'B' };
        private Program? program;

        private char stanGłowicy = 'q'; //qAAAB
        private int pozycjaGłowicy = 0;

        public void WczytajStan(string ścieżkaPliku)
        {
            string text = File.ReadAllText(ścieżkaPliku);
            pozycjaGłowicy = -1;
            for(int i = 0; i < text.Length; ++i)
            {
                char c = text[i];
                if(char.IsLower(c))
                {
                    if (pozycjaGłowicy >= 0) throw new TuringException("Błąd podczas wczytywania stanu. Powielona litera głowicy");
                    pozycjaGłowicy = i;
                    stanGłowicy = c;
                    text = ((i > 0) ? text.Substring(0, i - 1) : "") + ((i < text.Length) ? text.Substring(i + 1) : "");
                }
                if(pozycjaGłowicy < 0) throw new TuringException("Błąd podczas wczytywania stanu. Brak litery głowicy");
                taśma = text.ToCharArray().ToList();
            }
        }
        #endregion

        #region Nadzór
        private bool sprawdźPoprawnośćGłowicy()
        {
            if (!char.IsLower(stanGłowicy)) return false;
            if (pozycjaGłowicy < 0 || pozycjaGłowicy >= taśma.Count) return false;
            return true;
        }
        #endregion

        #region Procesor/Logika
        private void wykonajInstrukcję(Czwórka instrukcja)
        {
            switch (instrukcja.nzk)
            {
                case 'L':
                    if (pozycjaGłowicy - 1 < 0) throw new WyjściePozaTaśmęException();
                    pozycjaGłowicy--;
                    break;
                case 'R':
                    if (pozycjaGłowicy + 1 > taśma.Count - 1) throw new WyjściePozaTaśmęException();
                    pozycjaGłowicy++;
                    break;
                default:
                    taśma[pozycjaGłowicy] = instrukcja.nzk;
                    break;
            }
            stanGłowicy = instrukcja.nsg;
        }

        private List<string> historiaStanów = new List<string>(); //lepiej byłoby coś ze strukturą

        private bool wykryjZapętlenie()
        {
            for (int i = 1; i <= Math.Floor((double)historiaStanów.Count() / 2); i++)
            {
                int index = historiaStanów.Count() - i;
                List<string> list1 = historiaStanów.GetRange(index, i);
                List<string> list2 = historiaStanów.GetRange(index - i, i);
                if (Enumerable.SequenceEqual(list1, list2)) return true;
            }
            return false;
        }

        public void WykonajProgram()
        {
            #region Diagnostyka
            if (program == null) throw new BrakProgramuException();
            /*
            if (!sprawdźPoprawnośćProgramu(program))
            {
                throw new ProgramNieJestPoprawnyException();

            }
            */
            if (!sprawdźPoprawnośćGłowicy())
            {
                throw new StanGłowicyNieJestPoprawnyException();
            }
            #endregion

            bool czyKontynuować = true;            
            onStateChanged();
            int liczbaIteracjiZapętlenia = 0;
            do
            {                
                czyKontynuować = true;
                try
                {
                    Czwórka instrukcja = program.ZnajdźOdpowiedniąInstrukcję(stanGłowicy, taśma[pozycjaGłowicy]);
                    wykonajInstrukcję(instrukcja);
                    historiaStanów.Add(ToString());
                    if (wykryjZapętlenie())
                    {
                        dodajOstrzeżenie("Wykryto zapętlenie");
                        liczbaIteracjiZapętlenia++;
                        if (liczbaIteracjiZapętlenia > 2)
                        {
                            dodajOstrzeżenie("Działanie programu przerwane po wykryciu zapętlenia");
                            czyKontynuować = false;
                        }
                    }
                    onStateChanged(instrukcja.ToString());
                }
                catch (NieZnalezionoInstrukcjiException)
                {
                    dodajOstrzeżenie("Nie znaleziono instrukcji. Koniec programu");
                    czyKontynuować = false;
                }
                catch (WyjściePozaTaśmęException)
                {
                    dodajOstrzeżenie("Wyjście poza taśmę. Instrukcja nie została wykonana");
                }
                catch (TuringException)
                {
                    dodajOstrzeżenie("Inny błąd związany z działaniem maszyny Turingi");
                }
                catch (Exception exc)
                {
                    dodajOstrzeżenie("Inny błąd: " + exc.Message);
                }
            }
            while (czyKontynuować);            
        }
        #endregion

        public override string ToString()
        {
            return new string(taśma.ToArray()).Insert(pozycjaGłowicy, stanGłowicy.ToString());
        }        

        #region Zdarzenie
        public class StateChangedEventArgs : EventArgs
        {
            public string? OstatniaInstrukcja;
            public string? OpisStanu = null;
        }

        public event EventHandler<StateChangedEventArgs>? StateChanged;

        private void onStateChanged(string? ostatniaInstrukcja = null)
        {
            if (StateChanged != null)
            {
                string stan = this.ToString();
                StateChanged(this, new StateChangedEventArgs() { OpisStanu = stan, OstatniaInstrukcja = ostatniaInstrukcja });
            }
        }

        public event EventHandler? DodaneOstrzeżenie;

        private void onDodaneOstrzeżenie()
        {
            if (DodaneOstrzeżenie != null) DodaneOstrzeżenie(this, EventArgs.Empty);
        }
        #endregion
    }
}
