﻿using Microsoft.ML;
using Microsoft.ML.Data;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using static Microsoft.ML.DataOperationsCatalog;

namespace ConsoleApp2
{
    namespace SentimentAnalysis
    {
        class Program
        {
            static readonly string _dataPath = Path.Combine(Environment.CurrentDirectory, "Data", "C:\\Users\\Maciek\\Desktop\\yelp_labelled.txt");

            static void Main(string[] args)
            {
                // Utwórz kontekst/środowisko lokalne ML.NET — umożliwia dodanie kroków w celu zachowania wszystkiego razem
                // podczas odkrywania trenerów i transformacji ML.NET
                MLContext mlContext = new MLContext();

                TrainTestData splitDataView = LoadData(mlContext);

                ITransformer model = BuildAndTrainModel(mlContext, splitDataView.TrainSet);

                Evaluate(mlContext, model, splitDataView.TestSet);


                UseModelWithBatchItems(mlContext, model);

                Console.WriteLine();
                Console.WriteLine("=============== Koniec procesu ===============");
            }

            public static TrainTestData LoadData(MLContext mlContext)
            {
                // Zauważ, że w tym przypadku ładowanie danych treningowych z pliku
                // to najłatwiejszy sposób na rozpoczęcie pracy, ale ML.NET również na to pozwala
                // aby załadować dane z baz danych lub kolekcji w pamięci.

                IDataView dataView = mlContext.Data.LoadFromTextFile<SentimentData>(_dataPath, hasHeader: false);


                // Potrzebujesz zarówno uczącego zestawu danych do uczenia modelu, jak i testowego zestawu danych do oceny modelu.
                // Podziel załadowany zestaw danych na zestawy danych pociągowych i testowych
                // Określ procent zestawu danych testowych za pomocą parametru `testFraction`

                TrainTestData splitDataView = mlContext.Data.TrainTestSplit(dataView, testFraction: 0.2);

                return splitDataView;
            }

            public static ITransformer BuildAndTrainModel(MLContext mlContext, IDataView splitTrainSet)
            {
                // Utwórz elastyczny potok (składający się z łańcucha estymatorów) do tworzenia/trenowania modelu.
                // Służy do formatowania i czyszczenia danych.
                // Konwertuj kolumnę tekstową na wektory liczbowe (kolumna Cechy)

                var estimator = mlContext.Transforms.Text.FeaturizeText(outputColumnName: "Features", inputColumnName: nameof(SentimentData.SentimentText))

                // dołącz zadanie uczenia maszynowego do estymatora

                .Append(mlContext.BinaryClassification.Trainers.SdcaLogisticRegression(labelColumnName: "Label", featureColumnName: "Features"));


                // Utwórz i wytrenuj model na podstawie załadowanego i przekształconego zestawu danych.

                Console.WriteLine("=============== Utwórz i wytrenuj model ===============");
                var model = estimator.Fit(splitTrainSet);
                Console.WriteLine("=============== Koniec treningu ===============");
                Console.WriteLine();


                // Zwraca model, którego nauczyliśmy się używać do oceny.
                return model;

            }

            public static void Evaluate(MLContext mlContext, ITransformer model, IDataView splitTestSet)
            {
                // Oceń model i pokaż statystyki dokładności

                //Pobierz dane, dokonaj transformacji, wyślij dane.

                Console.WriteLine("=============== Ocena dokładności modelu za pomocą danych testowych===============");
                IDataView predictions = model.Transform(splitTestSet);


                // BinaryClassificationContext.Evaluate zwraca BinaryClassificationEvaluator.CalibratedResult
                // który zawiera obliczone ogólne metryki.

                CalibratedBinaryClassificationMetrics metrics = mlContext.BinaryClassification.Evaluate(predictions, "Label");


                // Metryka Accuracy pobiera dokładność modelu, czyli proporcję
                // poprawnych predykcji w zbiorze testowym.

                // Metryka AreaUnderROCCurve jest równa prawdopodobieństwu, że algorytm uszereguje się
                // losowo wybrana pozytywna instancja wyższa niż losowo wybrana negatywna
                // (zakładając, że „pozytywne” zajmuje wyższą pozycję niż „negatywne”).

                // Metryka F1Score pobiera wynik F1 modelu.
                // Wynik F1 jest średnią harmoniczną precyzji i pamięci:
                // 2 * precyzja * przywołanie / (precyzja + przywołanie).


                Console.WriteLine();
                Console.WriteLine("Ocena metryk jakości modelu");
                Console.WriteLine("--------------------------------");
                Console.WriteLine($"Precyzja: {metrics.Accuracy:P2}");
                Console.WriteLine($"Klasyfikacja: {metrics.AreaUnderRocCurve:P2}");
                Console.WriteLine($"F1Wynik: {metrics.F1Score:P2}");
                Console.WriteLine("=============== Koniec oceny ===============");

            }


            public static void UseModelWithBatchItems(MLContext mlContext, ITransformer model)
            {


                IEnumerable<SentimentData> sentiments = new[]
                {
                new SentimentData
                {
                    SentimentText = "Service sucks."
                },
                new SentimentData
                {
                    SentimentText = "I love this spaghetti."
                }
            };


                // Załaduj właśnie utworzone komentarze wsadowe
                IDataView batchComments = mlContext.Data.LoadFromEnumerable(sentiments);

                IDataView predictions = model.Transform(batchComments);

                // Użyj modelu, aby przewidzieć, czy dane komentarza są dodatnie (1), czy ujemne (0).

                IEnumerable<SentimentPrediction> predictedResults = mlContext.Data.CreateEnumerable<SentimentPrediction>(predictions, reuseRowObject: false);



                Console.WriteLine();

                Console.WriteLine("=============== Test predykcyjny załadowanego modelu z wieloma próbkami ===============");


                Console.WriteLine();


                foreach (SentimentPrediction prediction in predictedResults)
                {
                    Console.WriteLine($"Opinia: {prediction.SentimentText} | Prognoza: {(Convert.ToBoolean(prediction.Prediction) ? "Pozytywna" : "Negatywna")} | Prawdopodobienstwo: {prediction.Probability} ");
                }
                Console.WriteLine("=============== Koniec prognoz ===============");

            }
        }
    }

}
