﻿using Microsoft.EntityFrameworkCore;
using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace EF
{
    #region Klasy encji
    public class Osoba
    {
        [Key] public int Id { get; set; } //primary key
        public string Imię { get; set; }
        public string Nazwisko { get; set; }
        public int? NumerTelefonu { get; set; }

        public Adres Adres { get; set; }

        public override bool Equals(object? obj)
        {
            if (obj is not Osoba) return false;
            Osoba innaOsoba = obj as Osoba;
            return
                Id == innaOsoba.Id &&
                Imię.Equals(innaOsoba.Imię) &&
                Nazwisko.Equals(innaOsoba.Nazwisko);
        }

        public override int GetHashCode()
        {
            return Id ^ Imię.GetHashCode() ^ Nazwisko.GetHashCode() ^ NumerTelefonu.GetHashCode();
        }

        public override string ToString()
        {
            return $"({Id}) {Imię} {Nazwisko} ({(NumerTelefonu.HasValue ? NumerTelefonu.Value.ToString() : "---")}), {Adres}";
        }
    }

    public class Adres
    {
        [Key] public int Id { get; set; }
        public string Miasto { get; set; }
        public string Ulica { get; set; }
        public int NumerDomu { get; set; }
        public int? NumerMieszkania { get; set; }

        public override string ToString()
        {
            return $"({Id}) {Miasto}, {Ulica} {NumerDomu}/{NumerMieszkania}".TrimEnd('/');
        }

        public override bool Equals(object? obj)
        {
            if (obj is not Adres) return false;
            Adres innyAdres = obj as Adres;
            return
                Miasto.Equals(innyAdres.Miasto) &&
                Ulica.Equals(innyAdres.Ulica) &&
                NumerDomu == innyAdres.NumerDomu &&
                NumerMieszkania == innyAdres.NumerMieszkania;
        }
    }
    #endregion

    class BazaDanychContext : DbContext
    {
        public DbSet<Osoba> Osoby {  get; set; }
        public DbSet<Adres> Adresy { get; set; }

        protected override void OnConfiguring(DbContextOptionsBuilder optionsBuilder)
        {
            //base.OnConfiguring(optionsBuilder);
            optionsBuilder.UseSqlite("Data source=osoby.db");
        }
    }

    public class BazaDanychOsób : IDisposable
    {
        private BazaDanychContext dbc = new BazaDanychContext();

        public BazaDanychOsób()
        {
            dbc.Database.EnsureCreated();
        }

        public void Dispose()
        {
            dbc.Dispose();
        }

        public Osoba PobierzOsobę(int idOsoby)
        {
            return dbc.Osoby.FirstOrDefault(o => o.Id == idOsoby);
        }

        public Osoba this[int idOsoby]
        {
            get => PobierzOsobę(idOsoby);
            set => ZmieńDaneOsoby(idOsoby, value);
        }

        public int[] IdentyfikatoryOsób
        {
            get
            {
                return dbc.Osoby.Select(o =>  o.Id).ToArray();
            }
        }

        public int DodajOsobę(Osoba osoba)
        {
            if (osoba == null)
                throw new ArgumentNullException();
            if (osoba.Adres == null) throw new ArgumentException("W naszym kraju nie ma bezdomnych");

            if (dbc.Osoby.ToArray().Any(o => o.Equals(osoba))) return osoba.Id;
            else
            {
                if (dbc.Osoby.Any(o => o.Id == osoba.Id))
                    osoba.Id = dbc.Osoby.Max(o => o.Id) + 1;
            }

            Adres adres = dbc.Adresy.ToArray().FirstOrDefault(a => a.Equals(osoba.Adres));
            if (adres != null) osoba.Adres = adres;

            dbc.Osoby.Add(osoba);
            dbc.SaveChanges();

            return osoba.Id;
        }

#if DEBUG
        public Osoba[] Osoby { get { return dbc.Osoby.ToArray(); } }
        public Adres[] Adresy { get { return dbc.Adresy.ToArray(); } }
#endif

        #region Usuwanie
        private int[] pobierzIdentyfikatoryUżywanychAdresów()
        {
            return dbc.Osoby.Select(o => o.Adres.Id).Distinct().ToArray();
        }

        private Adres[] pobierzNieużywaneAdresy()
        {
            int[] używaneIdentyfikatoryAdresów = pobierzIdentyfikatoryUżywanychAdresów();
            List<Adres> nieużywaneAdresy = new List<Adres>();
            foreach(Adres adres in dbc.Adresy)
            {
                if (!używaneIdentyfikatoryAdresów.Contains(adres.Id))
                    nieużywaneAdresy.Add(adres);
            }
            return nieużywaneAdresy.ToArray();
        }

        private void usuńNieużywaneAdresy()
        {
            dbc.Adresy.RemoveRange(pobierzNieużywaneAdresy());
            dbc.SaveChanges();
        }

        public void UsuńOsobę(int idOsoby)
        {
            Osoba osoba = PobierzOsobę(idOsoby);
            if(osoba != null)
            {
                dbc.Osoby.Remove(osoba);
                dbc.SaveChanges();
                usuńNieużywaneAdresy();
            }
        }
        #endregion

        public void ZmieńDaneOsoby(int idOsoby, Osoba noweDane)
        {
            Osoba osoba = PobierzOsobę(idOsoby);
            if (osoba == null) throw new Exception("Nie ma osoby o podanym ID");
            osoba.Imię = noweDane.Imię;
            osoba.Nazwisko = noweDane.Nazwisko;
            osoba.NumerTelefonu = noweDane.NumerTelefonu;
            osoba.Adres = noweDane.Adres;
            dbc.SaveChanges();
        }
    }
}
