﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Threading.Tasks;

namespace MaszynaTuringa
{
    public class ProgramTuringa
    {
        #region Typy zagnieżdżone
        public enum BłądWKodzie
        {
            OK,
            CzwórkaNieMaCzterechZnaków,
            PierwszyZnakCzwórkiNieJestMałąLiterą,
            DrugiZnakCzwórkiNieJestDużąLiterą,
            TrzeciZnakCzwórkiNieJestDużąLiterą,
            CzwartyZnakCzwórkiNieJestMałąLiterą,
            ProgramNieJestJednoznaczny
        }

        public struct Błąd
        {
            public uint numerLinii;
            public BłądWKodzie rodzajBłędu;
        }

        public struct Czwórka
        {
            public char StanGłowicyZastany;
            public char WartośćNaTaśmieZastana;
            public char NowaWartośćNaTaśmieLubInstrukcja;
            public char NowyStanGłowicy;

            public static Czwórka Parsuj(string linia)
            {
                return new Czwórka()
                {
                    StanGłowicyZastany = linia[0],
                    WartośćNaTaśmieZastana = linia[1],
                    NowaWartośćNaTaśmieLubInstrukcja = linia[2],
                    NowyStanGłowicy = linia[3]
                };
            }

            public override string ToString()
            {
                return $"{StanGłowicyZastany}{WartośćNaTaśmieZastana}{NowaWartośćNaTaśmieLubInstrukcja}{NowyStanGłowicy}";
            }

            public static bool CzyCzwórkiOdpowiadająTemuSamemuStanu(Czwórka czwórka1, Czwórka czwórka2)
            {
                return (czwórka1.StanGłowicyZastany == czwórka2.StanGłowicyZastany) && (czwórka1.WartośćNaTaśmieZastana == czwórka2.WartośćNaTaśmieZastana);
            }
        }
        #endregion

        #region Stan
        private Czwórka[] program;
        private Błąd[] błędy;
        #endregion

        #region Akcesory
        //ta własność zwraca kopię
        public Błąd[] Błędy
        {
            get
            {
                return (Błąd[])błędy.Clone();
            }
        }

        public Czwórka[] Instrukcje => (Czwórka[])program.Clone();
        #endregion

        public ProgramTuringa(Czwórka[] program, Błąd[] błędy = null)
        {
            if (program == null) throw new ArgumentNullException("Brak obiektu programu");
            if (program.Length == 0) throw new ArithmeticException("Program nie zawiera żadnych linii poleń");
            this.program = program;
            this.błędy = błędy;
        }

        private static BłądWKodzie czyLiniaKoduJestPoprawna(string instrukcja)
        {
            if (instrukcja.Length != 4) return BłądWKodzie.CzwórkaNieMaCzterechZnaków;
            if (!char.IsLower(instrukcja[0])) return BłądWKodzie.PierwszyZnakCzwórkiNieJestMałąLiterą;
            if (!char.IsUpper(instrukcja[1])) return BłądWKodzie.DrugiZnakCzwórkiNieJestDużąLiterą;
            if (!char.IsUpper(instrukcja[2])) return BłądWKodzie.TrzeciZnakCzwórkiNieJestDużąLiterą;
            if (!char.IsLower(instrukcja[3])) return BłądWKodzie.CzwartyZnakCzwórkiNieJestMałąLiterą;
            return BłądWKodzie.OK;
        }

        private static bool czyProgramJednoznaczny(Czwórka[] instrukcje)
        {
            for (int i = 0; i < instrukcje.Length; ++i)
                for (int j = i + 1; j < instrukcje.Length; ++j)
                    if (Czwórka.CzyCzwórkiOdpowiadająTemuSamemuStanu(instrukcje[i], instrukcje[j])) return false;
            return true;
        }

        //pseudokonstruktor
        public static ProgramTuringa Wczytaj(string ścieżkaPliku)
        {
            try
            {
                string[] linie = System.IO.File.ReadAllLines(ścieżkaPliku);
                List<Błąd> błędy = new List<Błąd>();
                Czwórka[] instrukcje = new Czwórka[linie.Length];
                for (uint i = 0; i < linie.Length; ++i)
                {
                    BłądWKodzie wynikSprawdzenia = czyLiniaKoduJestPoprawna(linie[i]);
                    Błąd raportBłędu = new Błąd() { numerLinii = i + 1, rodzajBłędu = wynikSprawdzenia };
                    if (wynikSprawdzenia != BłądWKodzie.OK) błędy.Add(raportBłędu);
                    instrukcje[i] = Czwórka.Parsuj(linie[i]);
                }
                if (!czyProgramJednoznaczny(instrukcje))
                {
                    Błąd raportBłędu = new Błąd() { numerLinii = 0, rodzajBłędu = BłądWKodzie.ProgramNieJestJednoznaczny };
                    błędy.Add(raportBłędu);
                }
                return new ProgramTuringa(instrukcje, błędy.ToArray());
            }
            catch(Exception exc)
            {
                throw new TuringException("Błąd przy wczytywaniu programu", exc);
            }
        }

        public Czwórka? ZnajdźPasującąInstrukcję(char stanGłowicy, char wartośćNaTaśmie)
        {
            foreach(Czwórka instrukcja in Instrukcje)
            {
                if (instrukcja.StanGłowicyZastany == stanGłowicy && instrukcja.WartośćNaTaśmieZastana == wartośćNaTaśmie)
                    return instrukcja;
            }
            return null;
        }
    }

    public class StanMaszynyTuringa
    {
        public char StanGłowicy { get; private set; }
        public int PozycjaGłowicy { get; private set; }

        private StringBuilder taśma = new StringBuilder();        

        public string Taśma
        {
            get
            {
                return taśma.ToString();
            }
        }

        public char BieżącaWartośćNaTaśmie
        {
            get
            {
                return taśma[PozycjaGłowicy];
            }
            private set
            {
                taśma[PozycjaGłowicy] = value;
            }
        }

        private StanMaszynyTuringa() { } //to nie jest singleton, ale blokada możliwości utworzenia instancji inaczej niż przez pseudokonstruktor

        private static bool czyZapisPoprawny(string taśmaZGłowicą)
        {
            bool czyZnalezionoGłowicę = false;
            for (int i = 0; i < taśmaZGłowicą.Length; ++i)
            {
                char znak = taśmaZGłowicą[i];
                if (char.IsLower(znak))
                {
                    if (!czyZnalezionoGłowicę) czyZnalezionoGłowicę = true;
                    else return false;
                }
                if (!char.IsUpper(znak) && !char.IsLower(znak))
                    return false;
            }
            return true;
        }

        //TODO: zmienić nazwę
        private static StanMaszynyTuringa parsujTaśmęIGłowicę(string taśmaZGłowicą)
        {
            int pozycjaGłowicy = -1;
            for (int i = 0; i < taśmaZGłowicą.Length; ++i)
            {
                char znak = taśmaZGłowicą[i];
                if (char.IsLower(znak))
                {
                    pozycjaGłowicy = i;
                    break;
                }
            }

            StanMaszynyTuringa stan = new StanMaszynyTuringa()
            {
                StanGłowicy = taśmaZGłowicą[pozycjaGłowicy],
                PozycjaGłowicy = pozycjaGłowicy,
                taśma = new StringBuilder(taśmaZGłowicą.Remove(pozycjaGłowicy, 1))
            };
            return stan;
        }

        public static StanMaszynyTuringa Wczytaj(string ścieżkaPliku)
        {
            string taśmaZGłowicą = System.IO.File.ReadAllText(ścieżkaPliku);
            if (!czyZapisPoprawny(taśmaZGłowicą)) return null;
            return parsujTaśmęIGłowicę(taśmaZGłowicą);
        }

        public void Wykonaj(ProgramTuringa program)
        {
            do
            {
                ProgramTuringa.Czwórka? czwórka = program.ZnajdźPasującąInstrukcję(StanGłowicy, BieżącaWartośćNaTaśmie);
                if (!czwórka.HasValue) break;
                int poprzedniaPozycjaGłowicy = PozycjaGłowicy;
                char poprzedniStanGłowicy = StanGłowicy;
                StanGłowicy = czwórka.Value.NowyStanGłowicy;
                switch (czwórka.Value.NowaWartośćNaTaśmieLubInstrukcja)
                {
                    case 'L': PozycjaGłowicy--; break;
                    case 'R': PozycjaGłowicy++; break;
                    default: BieżącaWartośćNaTaśmie = czwórka.Value.NowaWartośćNaTaśmieLubInstrukcja; break;
                }
                #if DEBUG
                System.Threading.Thread.Sleep(100);
                #endif
                WykonanieInstrukcji(
                    this,
                    new WykonanieInstrukcjiEventArgs()
                    {
                        BieżącaPozycjaGłowica = PozycjaGłowicy,
                        BieżącyStanGłowicy = StanGłowicy,
                        PoprzedniaPozycjaGłowicy = poprzedniaPozycjaGłowicy,
                        PoprzedniStanGłowicy = poprzedniStanGłowicy,
                        Taśma = this.Taśma
                    });
            }
            while (true);
        }

        public string TaśmaZGłowicą
        {
            get
            {
                return Taśma.Insert(PozycjaGłowicy, StanGłowicy.ToString());
            }
        }

        #region Zdarzenie
        public class WykonanieInstrukcjiEventArgs : EventArgs
        {
            public int PoprzedniaPozycjaGłowicy, BieżącaPozycjaGłowica;
            public char PoprzedniStanGłowicy, BieżącyStanGłowicy;
            public string Taśma;
        }

        public event EventHandler<WykonanieInstrukcjiEventArgs> WykonanieInstrukcji;
        #endregion
    }

    #region Wyjątek
    public class TuringException : Exception
    {
        public TuringException(string message = null, Exception innerException = null)
            :base(message, innerException)
        { }
    }
    #endregion

    public class Turing
    {
        public ProgramTuringa Program { get; private set; }
        public StanMaszynyTuringa Stan { get; private set; }

        public Turing(string ścieżkaPlikuProgramu, string ścieżkaPlikuStanuMaszyny)
        {
            Program = ProgramTuringa.Wczytaj("program.txt");
            Stan = StanMaszynyTuringa.Wczytaj("taśma.txt");
        }

        public string Wykonaj()
        {
            if (Program.Błędy.Length == 0)
            {
                Stan.Wykonaj(Program);
                return Stan.TaśmaZGłowicą;
            }
            else throw new TuringException("Program posiada błędy. Wpierw je popraw, a potem spróbuj ponownie");            
        }

        //zwracam stan końcowy maszyny
        public Task<string> WykonajAsync()
        {
            return Task.Run(Wykonaj);            
        }
    }
}
