﻿using Microsoft.EntityFrameworkCore;
using System;
using System.Collections.Generic;
using System.Linq;

namespace EF
{
    public class BazaDanychOsóbDbContext : DbContext
    {
        public DbSet<Osoba> Osoby { get; set; }
        public DbSet<Adres> Adresy { get; set; }

        protected override void OnConfiguring(DbContextOptionsBuilder optionsBuilder)
        {
            optionsBuilder.UseSqlite("Data source=osoby.db");
        }
    }

    public class BazaDanychOsób : IDisposable
    {
        private BazaDanychOsóbDbContext dbc = new BazaDanychOsóbDbContext();

        public void Dispose()
        {
            dbc.Dispose();
        }

        public BazaDanychOsób()
        {
            dbc.Database.EnsureCreated();
        }

        public Osoba PobierzOsobę(int idOsoby)
        {
            return dbc.Osoby.FirstOrDefault(o => o.Id == idOsoby);
        }

        public Adres PobierzAdres(int idAdresu)
        {
            return dbc.Adresy.FirstOrDefault(a => a.Id == idAdresu);
        }

        //indekser
        public Osoba this[int idOsoby]
        {
            get => PobierzOsobę(idOsoby);
        }

        public int[] IdentyfikatoryOsób
        {
            get => dbc.Osoby.Select(o => o.Id).ToArray();
        }

        public int[] IdentyfikatoryAdresów
        {
            get => dbc.Adresy.Select(a => a.Id).ToArray();
        }

        public int DodajOsobę(Osoba osoba)
        {
            if (osoba == null) throw new ArgumentNullException("Null jest zły");
            if (osoba.Adres == null) throw new ArgumentException("W rejestrze nie można przechowywać osób bezdomnych");

            if (dbc.Osoby.ToArray().Any(o => o.Equals(osoba)))
                return osoba.Id;
            else
            {
                if (dbc.Osoby.Any(o => o.Id == osoba.Id))
                    osoba.Id = dbc.Osoby.Max(o => o.Id) + 1;
            }

            Adres adres = dbc.Adresy.ToArray().FirstOrDefault(a => a.Equals(osoba.Adres));
            if (adres != null) osoba.Adres = adres;

            dbc.Osoby.Add(osoba);
            dbc.SaveChanges();

            return osoba.Id;
        }

        private int[] pobierzIdentyfikatoryUżywanychAdresów()
        {
            return dbc.Osoby.Select(o => o.Adres.Id).Distinct().ToArray();
        }

        private Adres[] pobierzNieużywaneAdresy()
        {
            int[] identyfikatoryUżywanychAdresów = pobierzIdentyfikatoryUżywanychAdresów();
            List<Adres> nieużywaneAdresy = new List<Adres>();
            foreach(Adres adres in dbc.Adresy)
            {
                if (!identyfikatoryUżywanychAdresów.Contains(adres.Id))
                    nieużywaneAdresy.Add(adres);
            }
            return nieużywaneAdresy.ToArray();
        }

        private void usuńNieużywaneAdresy()
        {
            dbc.Adresy.RemoveRange(pobierzNieużywaneAdresy());
            dbc.SaveChanges();
        }

        public void UsuńOsobę(int idOsoby)
        {
            Osoba osoba = PobierzOsobę(idOsoby);
            if (osoba != null)
            {
                dbc.Osoby.Remove(osoba);
                dbc.SaveChanges();
                usuńNieużywaneAdresy();
            }
        }
    }
}
