﻿using System;
using System.IO;
using System.Collections.Generic;

namespace Turing
{
    //using static Console;
    using Czwórki = SortedList<
        (char bieżącyStanGłowicy, char bieżącaWartośćNaTaśmie), 
        (char nowyStanGłowicy, char nowaWartośćLubPolecenie)>;    

    class TuringException : Exception
    {
        public TuringException(string message = "", Exception innerException = null)
            :base(message, innerException)
        { }
    }

    public class Program
    {
        #region Łańcuch opisującym stan maszyny
        //public tylko ze względu na testy jednostkowe
        public static (char[] taśma, char stanGłowicy, int położenieGłowicy) analizujOpisStanuMaszyny(string łańcuchOpisującyStanMaszyny)
        {
            char początkowyStanGłowicy = ' ';
            int początkowePołożenieGłowicy = -1;
            string łańcuchTaśmy = "";
            for(int i = 0; i < łańcuchOpisującyStanMaszyny.Length; ++i)
            {
                char c = łańcuchOpisującyStanMaszyny[i];
                if (c == 'L' || c == 'R')
                    throw new TuringException("Litery L i R nie są dozwolone na taśmie");
                if(char.IsLower(c))
                {
                    if (początkowePołożenieGłowicy != -1)
                        throw new TuringException("Znaleziono więcej niż jeden znak oznaczający głowicę");
                    początkowyStanGłowicy = c;
                    początkowePołożenieGłowicy = i;
                    łańcuchTaśmy = łańcuchOpisującyStanMaszyny.Remove(i, 1);
                }
                else
                {
                    if (c < 'A' || c > 'Z')
                        throw new TuringException($"Niepoprawna wartość na taśmie {c} na pozycji {i}");
                }
            }
            if (początkowePołożenieGłowicy < 0)
                throw new TuringException("Nie znaleziono znaku oznaczającego głowicę");
            return (łańcuchTaśmy.ToCharArray(), początkowyStanGłowicy, początkowePołożenieGłowicy);
        }

        //public tylko ze względu na testy jednostkowe
        public static string pobierzŁańcuchOznaczającyStanMaszyny(
            (char[] taśma, char stanGłowicy, int położenieGłowicy) stanMaszyny)
        {
            string s = new string(stanMaszyny.taśma);
            s = s.Insert(stanMaszyny.położenieGłowicy, stanMaszyny.stanGłowicy.ToString());
            return s;
        }
        #endregion

        #region Parsowanie kodu programu
        static bool czyCzwórkaJestPoprawna(string linia)
        {
            Func<char, bool> isLower = (char c) => c >= 'a' && c <= 'z';
            Func<char, bool> isUpper = (char c) => c >= 'A' && c <= 'Z';

            if (linia.Length != 4) throw new TuringException("Nieodpowiednia liczba znaków w czwórce");
            //if (isLower(linia[0]) && isUpper(linia[1]) && isUpper(linia[2]) && isLower(linia[3])) return true;
            //else return false;
            //return (isLower(linia[0]) && isUpper(linia[1]) && isUpper(linia[2]) && isLower(linia[3])) ? true : false;
            return isLower(linia[0]) && isUpper(linia[1]) && isUpper(linia[2]) && isLower(linia[3]);
        }

        static Czwórki parsujProgram(string[] kodProgramu)
        {
            Czwórki czwórki = new Czwórki(kodProgramu.Length);
            foreach(string linia in kodProgramu)
            {
                if (string.IsNullOrWhiteSpace(linia)) continue;
                if (!czyCzwórkaJestPoprawna(linia)) throw new TuringException($"Niepoprawna linia kodu: {linia}");
                (char, char) stanBieżący = (linia[0], linia[1]);
                (char, char) nowyStan = (linia[3], linia[2]);
                if (czwórki.ContainsKey(stanBieżący)) throw new TuringException("Program nie może zawierać dwóch linii z tymi samymi dwoma literami");
                czwórki.Add(stanBieżący, nowyStan);
            }
            return czwórki;
        }
        #endregion

        //Nullable<typ> = typ?
        static (char nowyStanGłowicy, char nowaWartośćLubPolecenie)? znajdźPolecenie(char stanGłowicy, char wartośćNaTaśmie, Czwórki program)
        {
            (char stanGłowicy, char wartośćNaTaśmie) bieżącyStan = (stanGłowicy, wartośćNaTaśmie);
            if (program.ContainsKey(bieżącyStan)) return program[bieżącyStan];
            else return null;
        }

        static List<string> wykonajProgram(
            (char[] taśma, char stanGłowicy, int położenieGłowicy) stanMaszyny,
            Czwórki program)
        {
            List<string> wyjście = new List<string>();
            (char nowyStanGłowicy, char nowaWartośćLubPolecenie)? polecenie;
            while (
                (polecenie = znajdźPolecenie(
                    stanMaszyny.stanGłowicy,
                    stanMaszyny.taśma[stanMaszyny.położenieGłowicy],
                    program)) != null)
            {
                stanMaszyny.stanGłowicy = polecenie.Value.nowyStanGłowicy;
                switch(polecenie.Value.nowaWartośćLubPolecenie)
                {
                    case 'L':
                        stanMaszyny.położenieGłowicy--;
                        break;
                    case 'R':
                        stanMaszyny.położenieGłowicy++;
                        break;
                    default:
                        stanMaszyny.taśma[stanMaszyny.położenieGłowicy] = polecenie.Value.nowaWartośćLubPolecenie;
                        break;
                }
                wyjście.Add(pobierzŁańcuchOznaczającyStanMaszyny(stanMaszyny));
                //Console.WriteLine(pobierzŁańcuchOznaczającyStanMaszyny(stanMaszyny));
            }
            return wyjście;
        }

        static void Main(string[] args)
        {
            if(args.Length < 2)
            {
                Console.Error.WriteLine("Składnia: Turing program.txt taśma.txt");
                return;
            }

            string ścieżkaPlikuProgramu = args[0];
            if(!File.Exists(ścieżkaPlikuProgramu))
            {
                Console.Error.WriteLine($"Brak pliku programu {ścieżkaPlikuProgramu}");
                return;
            }

            string ścieżkaPlikuTaśmy = args[1];
            if(!File.Exists(ścieżkaPlikuTaśmy))
            {
                Console.Error.WriteLine($"Brak pliku taśmy {ścieżkaPlikuTaśmy}");
                return;
            }

            string[] kodProgramu = File.ReadAllLines(ścieżkaPlikuProgramu);
            string łańcuchOpisującyStanMaszyny = File.ReadAllText(ścieżkaPlikuTaśmy);

            Console.WriteLine("Początkowy stan maszyny: " + łańcuchOpisującyStanMaszyny);
            Console.WriteLine("Program: ");
            foreach (string linia in kodProgramu) Console.WriteLine(linia);

            try
            {
                var stanMaszyny = analizujOpisStanuMaszyny(łańcuchOpisującyStanMaszyny);
                Console.WriteLine("Stan głowicy: " + stanMaszyny.stanGłowicy);
                Console.WriteLine("Położenie głowicy: " + stanMaszyny.położenieGłowicy);
                Console.WriteLine("Taśma: " + new string(stanMaszyny.taśma));

                Czwórki program = parsujProgram(kodProgramu);
                foreach(KeyValuePair<(char, char), (char, char)> polecenie in program)
                {
                    Console.WriteLine(polecenie.ToString());
                }

                Console.WriteLine("Uruchomienie programu...");
                Console.WriteLine(pobierzŁańcuchOznaczającyStanMaszyny(stanMaszyny));
                List<string> wyjście = wykonajProgram(stanMaszyny, program);

                foreach (string liniaWyjścia in wyjście) Console.WriteLine(liniaWyjścia);
            }
            catch(TuringException exc)
            {
                Console.Error.WriteLine("Błąd maszyny Turinga: " + exc.Message);
            }
            catch(Exception exc)
            {
                Console.Error.WriteLine("Nierozpoznany błąd: " + exc.Message);
            }
        }
    }
}
