﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Reflection;
using Microsoft.Win32;

namespace MenedżerUstawień
{
    public class SettingsException : Exception
    {
        public SettingsException(string message, Exception innerException = null)
            : base(message, innerException)
        { }
    }

    public class SettingsManager<T>
        where T : class, new()
    {
        Dictionary<string, object> values = new Dictionary<string, object>();
        private string path;

        private SettingsManager(Dictionary<string, object> values, string path)
        {
            this.values = values; //tu tak naprawdę nie ma kopiowania
            this.path = path;
        }

        public SettingsManager(T settingsObject, string path)
            :this(extractValues(settingsObject),path)
        {
        }

        private static bool isProper(FieldInfo field)
        {
            return !field.IsNotSerialized && !field.IsStatic;
        }

        private static bool isProper(PropertyInfo property)
        {
            return property.CanRead && property.CanWrite;
        }

        private static Dictionary<string, object> extractValues(T settingsObject)
        {
            if (settingsObject != null)
            {
                Dictionary<string, object> values = new Dictionary<string, object>();

                //settingsObject.GetType()
                FieldInfo[] fields = typeof(T).GetFields();
                foreach (FieldInfo field in fields)
                {
                    if (isProper(field))
                    {
                        object instance = field.GetValue(settingsObject);
                        values.Add(field.Name, instance);
                    }
                }

                PropertyInfo[] properties = typeof(T).GetProperties();
                foreach (PropertyInfo property in properties)
                {
                    if (isProper(property))
                    {
                        object instance = property.GetValue(settingsObject);
                        values.Add(property.Name, instance);
                    }
                }
                return values;
            }
            else
            {
                throw new SettingsException("Object cannot be null");
            }
        }

        private static List<string> extractMembersNames()
        {
            List<string> memberNames = new List<string>();

            FieldInfo[] fields = typeof(T).GetFields();
            foreach (FieldInfo field in fields)
            {
                if (isProper(field))
                {
                    memberNames.Add(field.Name);
                }
            }

            PropertyInfo[] properties = typeof(T).GetProperties();
            foreach (PropertyInfo property in properties)
            {
                if (isProper(property))
                {
                    memberNames.Add(property.Name);
                }
            }

            return memberNames;
        }

        //--------------------------

        private static void saveToRegistry(Dictionary<string, object> values, string path)
        {
            RegistryKey key = Registry.CurrentUser.OpenSubKey(path, true);
            if (key == null) key = Registry.CurrentUser.CreateSubKey(path);
            foreach (KeyValuePair<string, object> value in values)
                key.SetValue(value.Key, value.Value);
            key.Close();
        }

        public void Save()
        {
            try
            {
                saveToRegistry(values, path);
            }
            catch (Exception exc)
            {
                throw new SettingsException("Settings saving error", exc);
            }
        }

        private static Dictionary<string, object> loadFromRegistry(List<string> memberNames, string path)
        {
            RegistryKey key = Registry.CurrentUser.OpenSubKey(path, false);
            if (key == null) throw new SettingsException("Registry key not found");
            Dictionary<string, object> values = new Dictionary<string, object>();
            foreach(string memberName in memberNames) values.Add(memberName, key.GetValue(memberName));
            key.Close();
            return values;
        }        

        public static SettingsManager<T> Load(string path)
        {
            if (true) //tu powinienem sprawdzić czy klucz istnieje
            {
                try
                {
                    List<string> memberNames = extractMembersNames();
                    Dictionary<string, object> values = loadFromRegistry(memberNames, path);
                    return new SettingsManager<T>(values, path);
                }
                catch (Exception exc)
                {
                    throw new SettingsException("Settings loading error", exc);
                }
            }
            else
            {
                //return new SettingsManager<T>(new T(), path);
                throw new SettingsException("No stored values found");                
            }
        }

        //--------------------------------------

        private static T buildObject(Dictionary<string, object> values)
        {
            T settingsObject = new T();
            
            FieldInfo[] fields = typeof(T).GetFields();
            foreach (FieldInfo field in fields)
            {
                if(isProper(field))
                {
                    object value = null;
                    if (values.Keys.Contains(field.Name)) value = values[field.Name];
                    else throw new SettingsException("No proper value found");
                    field.SetValue(settingsObject, value);
                }
            }

            PropertyInfo[] properties = typeof(T).GetProperties();
            foreach (PropertyInfo property in properties)
            {
                if (isProper(property))
                {
                    object value = null;
                    if (values.Keys.Contains(property.Name)) value = values[property.Name];
                    else throw new SettingsException("No proper value found");
                    property.SetValue(settingsObject, value);
                }
            }

            return settingsObject;
        }

        public T GetSettingsObject()
        {
            return buildObject(values);
        }
    }
}
