﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using System.IO;

//wersja 1.08 (2015-10-08)

namespace JacekMatulewski.Csv
{
    using Txt;

    public class CsvException : Exception 
    {
        public CsvException(string message)
            :base(message)
        {            
        }
    }

    public class CsvFormatException : CsvException 
    {
        public CsvFormatException(string message)
            :base(message)
        {
        }
    }
    public class CsvNoFilenameException : CsvException 
    {
        public CsvNoFilenameException(string message)
            :base(message)
        {
        }
    }

    public static partial class CsvExtensions
    {
        public static string Concat(this IEnumerable<string> list, char separatorChar)
        {
            string s = "";
            foreach(string element in list) s += element + separatorChar;
            return s;
        }

        public static string Concat<T>(this IEnumerable<T> list, char separatorChar)
        {
            string s = "";
            foreach (T element in list) s += element.ToString() + separatorChar;
            return s;
        }
    }    

    public interface ICsvRecord
    {
        //static ICsvRecord ParseLine(string line, char separator);
        //void ParseLine(string line, char separator, CsvRecordParam param);
        //string ToLine(char separator, CsvRecordParam param);

        void ParseValues(string[] values, CsvRecordParam param);
        string[] ToValues(CsvRecordParam param);
    }

    public class CsvRecordParam : Object //niepotrzebne, ale żeby zaznaczyć, że puste
    {
        public static readonly CsvRecordParam Empty;
    }

    public class CsvDocument<T> : IEnumerable<T>
        where T : ICsvRecord, new()
    {
        protected CsvRecordParam param;
        public const char CommentChar = '#';        
        protected Txt.ITextFile textFile = null; //przez dependency injection
        protected string headerComment = null;
        protected string headerColumnNames = null;
        protected char separatorChar;
        protected List<T> records = new List<T>();        

        public int Count
        {
            get
            {
                return records.Count;
            }
        }

        public ITextFile TextFile
        {
            get
            {
                return textFile;
            }
        }

        public string HeaderComment
        {
            get
            {
                return headerComment;
            }
        }

        public string HeaderColumnNames
        {
            get
            {
                return headerColumnNames;
            }
        }

        public char SeparatorChar
        {
            get
            {
                return separatorChar;
            }
        }

        public T this[int index]
        {
            get
            {
                return records[index];
            }
        }

        public T[] GetRecords()
        {
            return records.ToArray();
        }

        public CsvDocument(CsvRecordParam param = null) 
        {
            this.param = param;
        }

        //copyData jest na wyrost, bo robi tylko nową listę z tymi samymi referencjami
        //można dla ICloneable: if (record is ICloneable) _record = (T)((ICloneable)record).Clone(); else _record = record;
        public CsvDocument(List<T> records, CsvRecordParam param = null, bool copyData = false) 
        {
            this.param = param;
            if (!copyData) this.records = records;
            else foreach (T record in records) AddRecord(record, false); 
        }

        //na potrzeby klas potomnych, udostępniana w funkcji Load
        protected CsvDocument(ITextFile textFile, char separatorChar, CsvRecordParam param = null)
        {
            //load lines from text file
            string[] lineList = textFile.Load();

            //create instance
            this.param = param;
            this.textFile = textFile;
            this.separatorChar = separatorChar;
            List<string> comments = new List<string>();

            //parse lines and build CsvDocument
            foreach (string line in lineList)
            {
                if (!line.StartsWith(CommentChar.ToString()))
                {
                    T record = new T();
                    //record.ParseLine(line, separatorChar);
                    if (!TryParseRecordSecured(ref record, line, separatorChar, param))
                        throw new CsvFormatException("Parsing line error: " + line);
                    else this.records.Add(record); //dodaję bezpośrednio, a nie przez AddRecord(record, false)
                }
                else comments.Add(line);
            }
            switch (comments.Count)
            {
                case 0:
                    this.headerComment = null;
                    this.headerColumnNames = null;
                    break;
                case 1:
                    this.headerComment = null;
                    this.headerColumnNames = comments[0];
                    break;
                default:
                    this.headerComment = comments[0];
                    this.headerColumnNames = comments[1];
                    break;
            }
        }

        public static CsvDocument<T> CreateEmptyFile(ITextFile textFile, char separatorChar, string headerComment, string headerColumnNames, CsvRecordParam param = null)
        {
            CsvDocument<T> csv = new CsvDocument<T>(param);
            csv.textFile = textFile;
            csv.SaveAs(textFile, separatorChar, headerComment, headerColumnNames);
            return csv;
        }

        private static bool TryParseRecordSecured(ref T record, string line, char separatorChar, CsvRecordParam param)
        {
            try
            {
                line = line.Replace("^R", "\r"); //powrót karetki
                line = line.Replace("^N", "\n"); //nowe linie                
                //string[] strings = line.Trim(separatorChar).Split(separatorChar);
                string[] strings = line.Split(separatorChar);
                record.ParseValues(strings, param);
                return true;
            }
            catch//(Exception exc)
            {                
                return false;
            }
        }      

        public static CsvDocument<T> Load(ITextFile textFile, char separatorChar, CsvRecordParam param = null)
        {
            return new CsvDocument<T>(textFile, separatorChar, param);
        }

        private static string ValuesToLine(string[] values, char separatorChar)
        {
            return values.Concat(separatorChar);
        }

        private static string RecordToLineSecured(ICsvRecord record, char separatorChar, CsvRecordParam param)
        {
            string line = ValuesToLine(record.ToValues(param), separatorChar);
            line = line.Replace('^', '!'); //usuwam znak używany do kodowania znaków specjalnych
            line = line.Replace("\r", "^R"); //powrót karetki
            line = line.Replace("\n", "^N"); //nowe linie
            return line;
        }

        public void SaveAs(ITextFile textFile, char separatorChar, string headerComment = null, string headerColumnNames = null)
        {
            //nadpisywanie
            this.separatorChar = separatorChar;
            if (headerComment != null) this.headerComment = headerComment;
            if (headerColumnNames != null) this.headerColumnNames = headerColumnNames;
            if (textFile != null) this.textFile = textFile;
            
            //zapis
            /*
            using (StreamWriter sw = new StreamWriter(filename, false, Encoding.UTF8))
            {
                if (!string.IsNullOrEmpty(headerComment)) sw.WriteLine(CommentChar + headerComment);
                if (!string.IsNullOrEmpty(headerColumnNames)) sw.WriteLine(CommentChar + headerColumnNames);
                foreach (ICsvRecord record in records)
                {
                    sw.WriteLine(RecordToLineSecured(record, separatorChar, param));
                }
            }
            */
            List<string> lineList = new List<string>();
            if (!string.IsNullOrEmpty(headerComment)) lineList.Add(CommentChar + headerComment);
            if (!string.IsNullOrEmpty(headerColumnNames)) lineList.Add(CommentChar + headerColumnNames);
            foreach (ICsvRecord record in records)
            {
                lineList.Add(RecordToLineSecured(record, separatorChar, param));
            }
            textFile.Save(lineList.ToArray());
        }

        private void AddLineToFile(string line)
        {
            /*
            using (StreamWriter sw = new StreamWriter(filename, true, Encoding.UTF8))
            {
                if(!string.IsNullOrWhiteSpace(line))
                    sw.WriteLine(line);
            }
            */
            if (!string.IsNullOrWhiteSpace(line))
                textFile.AppendLine(line);
        }

        protected void AddRecordToFile(T newRecord)
        {
            //AddLineToFile(newRecord.ToLine(separatorChar));
            AddLineToFile(RecordToLineSecured(newRecord, separatorChar, param));
        }

        private void AddCommentToFile(string comment)
        {
            AddLineToFile(comment);
        }

        public void AddRecord(T newRecord, bool saveToFile = true)
        {
            records.Add(newRecord); //nie robi klonów, można by: if(record is ICloneable) records.Add(newRecord.Clone());
            if(saveToFile) AddRecordToFile(newRecord);
        }

        public void RemoveRecord(T record, bool saveToFile = true)
        {
            records.Remove(record);
            if (saveToFile) Save();
        }

        public void RemoveRecord(int index, bool saveToFile = true)
        {
            records.RemoveAt(index);
            if (saveToFile) Save();
        }

        public void UpdateRecord(int index, T newRecord, bool saveToFile = true)
        {
            records[index] = newRecord;
            if (saveToFile) Save();
        }

        public void UpdateRecord(T record, T newRecord, bool saveToFile = true)
        {
            int index = records.FindIndex(r => r.Equals(record) );
            if (index < 0) throw new CsvException("Record not found");
            UpdateRecord(index, newRecord, saveToFile);
        }

        public void Save()
        {
            if (textFile == null) throw new CsvNoFilenameException("No file specified");
            SaveAs(textFile, separatorChar, headerComment, headerColumnNames);
        }

        public List<T> ToList()
        {
            //kopiuje dane
            return records.ToList();
        }

        public IEnumerator<T> GetEnumerator()
        {
            return records.GetEnumerator();
        }

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return this.GetEnumerator();
        }
    }
}
