﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace JacekMatulewski.Csv_KursNet
{
    public class CsvRecordParam : Object
    {
        public static readonly CsvRecordParam Empty;
    }

    public interface ICsvRecord
    {
        void ParseValues(string[] values, CsvRecordParam param);
        string[] ToValues(CsvRecordParam param);
    }

    public class CsvDocument<T> : IEnumerable<T>
        where T : ICsvRecord, new()
    {
        protected CsvRecordParam param;
        public static char CommentChar = '#';
        protected string filename = null;
        private string headerComment = null;
        protected string headerColumnNames = null;
        protected char separatorChar;
        protected List<T> records = new List<T>();

        public int Count
        {
            get
            {                
                return records.Count;
            }
        }

        public string Filename
        {
            get
            {
                return filename;
            }
        }

        protected string HeaderComment
        {
            get 
            { 
                return headerComment; 
            }
        }

        protected string HeaderColumnNames
        {
            get
            {
                return headerColumnNames;
            }
        }

        public char SeparatorChar
        {
            get
            {
                return separatorChar;
            }
        }

        public T this[int index] //indexer
        {
            get
            {
                return records[index];
            }
        }

        public T[] GetRecords()
        {
            return records.ToArray(); //robi kopię rekordów
        }

        public CsvDocument(CsvRecordParam param = null)
        {
            this.param = param;
        }

        public CsvDocument(List<T> records, CsvRecordParam param = null, bool copyData = false)
        {
            this.param = param;
            if (!copyData) this.records = records;
            else foreach (T record in records) AddRecord(record, false);
        }

        public static CsvDocument<T> CreateEmptyFile(string filename, char separatorChar, string headerComment, string headerColumnNames, CsvRecordParam param = null)
        {
            CsvDocument<T> csv = new CsvDocument<T>(param);
            csv.filename = filename;
            csv.SaveAs(filename, separatorChar, headerComment, headerColumnNames);
            return csv;
        }

        private static bool TryParseRecordSecured(ref T record, string line, char separatorChar, CsvRecordParam param)
        {
            try
            {
                line.Replace("^R", "\r");
                line.Replace("^N", "\n");
                string[] strings = line.Split(separatorChar);
                record.ParseValues(strings, param);
                return true;
            }
            catch
            {
                return false;
            }
        }

        public static CsvDocument<T> Load(string filename, char separatorChar, CsvRecordParam param = null)
        {
            CsvDocument<T> csv = new CsvDocument<T>(param);
            csv.filename = filename;
            csv.separatorChar = separatorChar;
            List<string> comments = new List<string>();
            using (StreamReader sr = new StreamReader(filename))
            {
                string line;
                while ((line = sr.ReadLine()) != null)
                {
                    if (!line.StartsWith(CommentChar.ToString()))
                    {
                        T record = new T();
                        if (!TryParseRecordSecured(ref record, line, separatorChar, param))
                            throw new Exception("Nie udało się odczytać linii: " + line);
                        else csv.AddRecord(record, false);
                    }
                    else comments.Add(line);
                }
            }
            switch(comments.Count)
            {
                case 0:
                    csv.headerComment = null;
                    csv.headerColumnNames = null;
                    break;
                case 1:
                    csv.headerComment = null;
                    csv.headerColumnNames = comments[0];
                    break;
                default:
                    csv.headerComment = comments[0];
                    csv.headerColumnNames = comments[1];
                    break;
            }
            return csv;
        }

        public void SaveAs(string filename, char separatorChar, string headerComment = null, string headerColumnNames = null)
        {
            this.separatorChar = separatorChar;
            this.headerComment = headerComment;
            this.headerColumnNames = headerColumnNames;
            this.filename = filename;
            using(StreamWriter sw = new StreamWriter(filename, false, Encoding.UTF8))
            {
                if(!string.IsNullOrWhiteSpace(headerComment)) sw.WriteLine(CommentChar + headerComment);
                if (!string.IsNullOrWhiteSpace(headerColumnNames)) sw.WriteLine(CommentChar + headerColumnNames);
                foreach(ICsvRecord record in records)
                {
                    sw.WriteLine(RecordToLineSecured(record, separatorChar, param));
                }
            }
        }

        private static string ValuesToLine(string[] values, char separatorChar)
        {
            string line = "";
            foreach (string value in values) line += value + separatorChar;
            return line;
        }

        private static string RecordToLineSecured(ICsvRecord record, char separatorChar, CsvRecordParam param)
        {
            string line = ValuesToLine(record.ToValues(param), separatorChar);
            line = line.Replace('^', '!'); //usuwam znak kodowania znaków specjalnych
            line = line.Replace("\r", "^R");
            line = line.Replace("\n", "^N");
            return line;
        }

        private void AddLineToFile(string line)
        {
            using (StreamWriter sw = new StreamWriter(filename, true, Encoding.UTF8))
            {
                if (!string.IsNullOrWhiteSpace(line)) sw.WriteLine(line);
            }
        }

        protected void AddRecordToFile(T record)
        {
            AddLineToFile(RecordToLineSecured(record, separatorChar, param));
        }

        public void AddRecord(T newRecord, bool saveToFile = true)
        {
            records.Add(newRecord);
            if (saveToFile) AddRecordToFile(newRecord);
        }

        public void AddComment(string comment)
        {
            AddLineToFile(comment);
        }


        /*
        TODO: Update, Remove, ToList
        */

        public void Save()
        {
            if (string.IsNullOrWhiteSpace(filename)) throw new FileNotFoundException("No filename specified");
            else SaveAs(filename, separatorChar, headerComment, headerColumnNames);
        }

        public IEnumerator<T> GetEnumerator()
        {
            return records.GetEnumerator();
        }

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return this.GetEnumerator();
        }
    }
}
