﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Input;

namespace PESELValidator.Model
{
	// Kobieta - 0,2,4,8 (nr % 2 == 0)
	// Mężczyzna - 1,3,5,7,9 (nr % 2 == 1)
	enum Płeć : byte
	{
		Kobieta = 0,
		Mężczyzna = 1
	}

	enum Miesiąc : byte
	{
		Styczeń		= 1,
		Luty		= 2,
		Marzec		= 3, 
		Kwiecień	= 4,
		Maj			= 5,
		Czerwiec	= 6, 
		Lipiec		= 7, 
		Sieprień	= 8,
		Wrzesień	= 9,
		Październik	= 10,
		Listopad	= 11,
		Grudzień	= 12
	}

	class PESEL
	{
		/// <summary>Składowe numeru Pesel posortowane według cyft</summary>
		//								   nr cyfr
		private int yearRaw;	// rok		[1,2]
		private int monthCode;	// miesiąc	[3,4]
		private int dayCode;	// dzień	[5,6]

		private int seriesNr;	// seria	[7,9]
		private int sexNr;		// płeć		[10]
		private int ctrlDigit;	// cyfra kontrolna [11]

		/// <summary>Przekazuje string z informacją o błędzie</summary>
		public event Action<string> ModelError = delegate { };

		/// <summary>Rok - części dziesiętne. Dozwolone wartośći [0,99]</summary>
		public int YearCode
		{
			get {
				if ( yearRaw >= 1800 )
					return int.Parse( yearRaw.ToString().Substring( 2, 2 ) );
				else
					return 0;
			}
			set {
				if ( value >= 0 && value < 100 )
				{ yearRaw /= 100; yearRaw = yearRaw * 100 + value; }
				else
					ModelError( " Wartość YearCode != [0,100]" );
			}
		}
		/// <summary>Rok w pełnym formacie </summary>
		public int Year
		{
			get { return yearRaw; }
			set { if ( ValidateYear( value ) )
					yearRaw = value;
				else
					ModelError( "Rok z poza zakresu 1800-2299" );
			}
		}

		/// <summary>Miesiąc w sformatowanej formie zależnej od roku</summary>
		public int Month
		{
			get { return monthCode; }
			set { if ( ValidateMonth( value ) )
				{
					yearRaw = YearFromMonth( value ) + YearCode;
					monthCode = value % 13;
				}
				else
					ModelError( "Miesiąc z poza zakresu 1-12" );
			}
		}

		/// <summary>Kod miesiąca w numerze PESEL</summary>
		public int MonthCode
		{
			get { return monthCode + AddToMonth(yearRaw); }
		}

		/// <summary>Dzień miesiąca</summary>
		public int Day
		{
			get { return dayCode; }
			set { if ( ValidateDays( value, monthCode, yearRaw ) )
					dayCode = value;
				else
					ModelError( "Dzień z poza zakresu dla danego miesiąca" );
			}
		}

		/// <summary>Generowany losowo numer serii numeru PESEL</summary>
		public int SeriesNr
		{
			get { return seriesNr; }
			set { seriesNr = value; }
		}

		/// <summary>Płeć wygenerowana na podstawie 10-tej cyfry</summary>
		public Płeć Sex
		{
			get { return sexNr % 2 == 1 ? Płeć.Mężczyzna : Płeć.Kobieta; }
		}

		public PESEL(Action<string> OnModelErrorAction = null)
		{
			ModelError += OnModelErrorAction;

			Month = 1;
			Year = 1900;
			Day = 1;

			SeriesNr = 0;
			sexNr = 0;
			ctrlDigit = 0;
		}

		public PESEL(string peselStr, Action<string> OnModelErrorAction = null)
		{
			ModelError += OnModelErrorAction;

			if (peselStr.Length != 11)
				ModelError( "Długość numeru PESEL różna od 11" );

			Month = int.Parse( peselStr.Substring( 2, 2 ) );
			YearCode = int.Parse( peselStr.Substring( 0, 2 ) );
			Day = int.Parse( peselStr.Substring( 4, 2 ) );
			SeriesNr = int.Parse( peselStr.Substring( 6, 3 ) );

			sexNr = int.Parse( peselStr.Substring( 9, 1 ) );
			ctrlDigit = int.Parse( peselStr.Substring( 10, 1 ) );
		}


		public PESEL(int year, int month, int day, Płeć sex, Action<string> OnModelErrorAction = null )
		{
			ModelError += OnModelErrorAction;

			Year = year;
			Month = month;
			Day = day;

			sexNr = (byte)sex;

		}

		public static bool ValidateYear(int year)
		{
			return (year >= 1800 && year <= 2299);
		}

		public static bool ValidateMonth(int m)
		{
			return (m >= 1 && m <= 12 ||
					m >= 21 && m <= 32 ||
					m >= 41 && m <= 52 ||
					m >= 61 && m <= 72 ||
					m >= 81 && m <= 92);
		}

		public static bool ValidateDays(int days, int month, int year)
		{
			int validDays = DateTime.DaysInMonth( year, month );

			return (days > 0 && days <= validDays);
		}

		/// <summary>Zwraca określoną liczbę do dodania do kodu miesiąca zależną od roku</summary>
		/// <returns>liczba do dodania</returns>
		public int AddToMonth( int year )
		{
			if ( year >= 1800 && year <= 1899 )
				return 80;
			else if ( year >= 1900 && year <= 1999 )
				return 0;
			else if ( year >= 2000 && year <= 2099 )
				return 20;
			else if ( year >= 2100 && year <= 2199 )
				return 40;
			else if ( year >= 2200 && year <= 2299 )
				return 60;

			ModelError( "Rok z poza zakresu 1800-2299" );
			return 0;
		}

		/// <summary>Zwraca n-setne tysiąclecie na podstawie kodu miesięca</summary>
		/// <param name="month"></param>
		/// <returns></returns>
		public int YearFromMonth( int month )
		{
			if ( month >= 81 && month <= 92 )
				return 1800;
			else if ( month >= 1 && month <= 12 )
				return 1900;
			else if ( month >= 21 && month <= 32 )
				return 2000;
			else if ( month >= 41 && month <= 52 )
				return 2100;
			else if ( month >= 61 && month <= 72 )
				return 2200;

			ModelError( "Kod miesiąca nie jest prawidłowy" );
			return 0;	
		}

		/// <summary>Sprawdza, czy pierwsze 10 cyfr generuje 11 cyfrę kontrolną</summary>
		/// <param name="p">Typ Pesel do sprawdzenia</param>
		/// <returns>Zgodność numeru PESEL</returns>
		public bool Validate(PESEL pesel)
		{
			if ( pesel == null )
				return false;

			string pstr = pesel.ToString();
			if ( pstr.Length != 11 )
				ModelError( "Długość numeru PESEL różna od 11" );

			if ( PESEL.ControlDigit( pstr ) == int.Parse( pstr[10].ToString() ) )
				return true;
			else
				return false;
		}

		public bool Validate(string pesel)
		{
			if ( pesel == null )
				return false;
			if ( pesel.Length != 11 )
				ModelError( "Długość numeru PESEL różna od 11" );

			if ( PESEL.ControlDigit( pesel ) == int.Parse( pesel[10].ToString() ) )
				return true;
			else
				return false;
		}

		/// <summary> Wyznacza cyfrę kontrolą na podstawie pierwszych 10 cyfr </summary>
		/// <param name="first10Digits">pierwsze 10 cyfr</param>
		/// <returns>Wyliczoną cyfrę kontrolną [0-9]</returns>
		public static int ControlDigit( string first10Digits )
		{
			int sum = 0;
			string s = first10Digits;

			sum += int.Parse( s[0].ToString() ) * 1;
			sum += int.Parse( s[1].ToString() ) * 3;
			sum += int.Parse( s[2].ToString() ) * 7;
			sum += int.Parse( s[3].ToString() ) * 9;
			sum += int.Parse( s[4].ToString() ) * 1;
			sum += int.Parse( s[5].ToString() ) * 3;
			sum += int.Parse( s[6].ToString() ) * 7;
			sum += int.Parse( s[7].ToString() ) * 9;
			sum += int.Parse( s[8].ToString() ) * 1;
			sum += int.Parse( s[9].ToString() ) * 3;

			return sum % 10;
		}

		public override string ToString()
		{
			string
				day = dayCode.ToString().Substring( 0, 2 ),
				month = monthCode.ToString().Substring( 0, 2 ),
				year = yearRaw.ToString().Substring( 2, 2 ),
				series = seriesNr.ToString().Substring( 0, 3 ),
				sex = sexNr.ToString().Substring( 0, 1 );

			string
				pstr = day + month + year + series + sex;
			pstr = pstr + PESEL.ControlDigit(pstr);

			return pstr;
		}

	}
}
