﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.ComponentModel;
using System.Windows.Input;
using System.Windows.Controls;
using System.Windows;

using MultipTabForKids.Model;
using MultipTabForKids.View;
using System.Collections.ObjectModel;

namespace MultipTabForKids.ViewModel
{
	/// <summary>Tryb widoku pytania w Quizie</summary>
	public enum QuestionViewMode : byte
	{
		ABC,
		Text
	}

	/// <summary>Widok pozwalający na sterowanie parametrami Quizu i ładowaniem widoków pytań do niego</summary>
	public class MulTabViewViewModel : INotifyPropertyChanged
	{
		/// <summary>Kolekcja pytań w aktualnie wybranym trybie </summary>
		ObservableCollection<object> obsQuestionList;

		QuestionViewMode questionListMode;	// tryb Quizu
		
		int questionsCount;				// ilość pytań
		int goodAnswersCount;			// ilość poprawnych odpowiedzi
		int factorCount;				// ilość mnożonych liczb
		string quizResult = "Wynik: ";
		bool isEditable = true;

		public MulTabViewViewModel()
		{
			QuestionsCount		= 10;
			goodAnswersCount	= 0;
			factorCount			= 2;
			obsQuestionList		= new ObservableCollection<object>();
			questionListMode	= QuestionViewMode.ABC;

			genQuestionsCmd			= new RelayCommand( GenerateQuestions, CanGenerateQuestions );
			checkQuestionsCmd		= new RelayCommand( CheckQuestions, CanCheckQuestions );
			changeQuestionsModeCmd	= new RelayCommand( ChangeQuestionMode );
			resetViewCmd			= new RelayCommand( ResetView );

			//obsQuestionList.Add( new ABCQuestionViewModel( 1, MulTab.Generate() ) );
		}

		ICommand genQuestionsCmd;
		/// <summary>Generowanie pytań</summary>
		public ICommand GenerateQuestionCommand
		{ get { return genQuestionsCmd; } }

		private void GenerateQuestions( object arg )
		{
			var q = MulTabGenerator.GenerateList( questionsCount, factorCount );

			for ( int i = 0 ;i < questionsCount ;i++ )
			{
				if ( questionListMode == QuestionViewMode.ABC )
					obsQuestionList.Add( new ABCQuestionViewModel( i, q[i] ) );
				else if ( questionListMode == QuestionViewMode.Text )
					obsQuestionList.Add( new TextQuestionViewModel( i, q[i] ) );
			}
			OnPropertyChanged( "IsGenerated" );
			q = null;
		}

		private bool CanGenerateQuestions( object arg )
		{
			return obsQuestionList.Count == 0;
		}

		ICommand checkQuestionsCmd;
		/// <summary>Sprawdzanie pytań</summary>
		public ICommand CheckQuestionsCommand
		{ get { return checkQuestionsCmd; } }

		private bool CanCheckQuestions( object arg )
		{
			foreach (ICheckAnswer question in obsQuestionList)
				if ( question.IsAnswered == false )
					return false;
			return true;
		}

		private void CheckQuestions( object arg )
		{
			foreach ( ICheckAnswer question in obsQuestionList )
				if ( question.CheckAnswer() == true )
					goodAnswersCount++;

			//IsEditable = false;
			QuizResult = "Wynik: " + goodAnswersCount + " / " + questionsCount;
		}

		ICommand changeQuestionsModeCmd;
		/// <summary>Zmiana trybu pytań</summary>
		public ICommand ChangeQuestionsModeCommand
		{ get { return changeQuestionsModeCmd; } }

		ICommand resetViewCmd;
		/// <summary>Wyczyszczenie listy pytań</summary>
		public ICommand ResetViewCommand
		{ get { return resetViewCmd; } }

		/// <summary>Wynik Quizu</summary>
		public string QuizResult
		{
			get { return quizResult; }
			set { quizResult = value; OnPropertyChanged( "QuizResult" ); }
		}

		/// <summary>Czy wygenerowano pytania</summary>
		public bool IsGenerated
		{ get { return obsQuestionList.Count > 0; } }

		/// <summary>Czy pytania można jeszcze edytować</summary>
		public bool IsEditable
		{
			get { return isEditable; }
			set { isEditable = value; OnPropertyChanged( "IsEditable" ); }
		}

		public int QuestionsCount
		{
			get { return questionsCount; }
			set { if ( value >= 0 )
				{ questionsCount = value; OnPropertyChanged( "QuestionsCount" ); } }
		}

		public ObservableCollection<object> QuestionsList
		{ get { return obsQuestionList; } }

		// Powinna być niedostępna blokowana, gdy lista wygenerowana
		// lista musi być czysta przed wygenerowaniem
		public QuestionViewMode QuestionMode
		{
			get { return questionListMode; }
			set { questionListMode = value; OnPropertyChanged( "QuestionMode" ); }
		}

		private void ResetView( object arg )
		{
			MulTabView mtv = arg as MulTabView;

			goodAnswersCount = 0;
			obsQuestionList.Clear();
			mtv.lblResult.Content = "Wynik: ";
			TextQuestionViewModel.TabIndex = 1;

			//IsEditable = true;
			OnPropertyChanged( "IsGenerated" );
		}

		private void ChangeQuestionMode(object arg)
		{
			MulTabView mtv = arg as MulTabView;

			switch ( QuestionMode )
			{
			case QuestionViewMode.ABC:
				mtv.lboxQuestions.ItemTemplate = ABCQuestionViewModel.QuestionTemplate;
				break;
			case QuestionViewMode.Text:
				mtv.lboxQuestions.ItemTemplate = TextQuestionViewModel.QuestionTemplate;
				break;
			}
		}

		public event PropertyChangedEventHandler PropertyChanged = delegate { };

		private void OnPropertyChanged( params string[] properties )
		{
			foreach ( string prop in properties )
				PropertyChanged( this, new PropertyChangedEventArgs( prop ) );
		}
	}//MulTabViewViewModel
}
