﻿using System;
using System.Threading.Tasks;

using System.Numerics;

namespace JacekMatulewski
{
    public class ZbioryFraktalne
    {
        public static Complex DefaultMin
        {
            get
            {
                return new Complex(-2.2, -1.2);
            }
        }
        public static Complex DefaultMax
        {
            get
            {
                return new Complex(1.2, 1.2);
            }
        }

        private const byte N = 255; //ilosc iteracji				        

        //min,max - zakres badanej plaszczyzny zespolonej
        public Complex Min { set; get; }
        public Complex Max { set; get; }
        public Complex Rozmiar
        {
            get
            {
                return Max - Min;
            }
        }

        public int Szerokosc { set; get; } //auto-implemented properties
        public int Wysokosc { set; get; }

        public ZbioryFraktalne(int szerokosc, int wysokosc, Complex min, Complex max)
        {
            this.Szerokosc = szerokosc;
            this.Wysokosc = wysokosc;
            this.Min = min;
            this.Max = max;
        }

        public Complex ClientToZespolone(int x, int y)
        {
            return Min + new Complex(x * (Max.Real - Min.Real) / Szerokosc, y * (Max.Imaginary - Min.Imaginary) / Wysokosc);
        }        

        public byte[,] ZbiorMandelbrota
        {
            get
            {
                return Oblicz(true);
            }
        }

        public byte[,] ZbiorJulii
        {
            get
            {
                return Oblicz(false);
            }
        }

        private byte[,] Oblicz(bool zbiorMandelbrotaCzyJulii = true)
        {
            if (Szerokosc <= 0 || Wysokosc <= 0) throw new ArgumentException("Szerokość i wysokość zbioru nie mogą być równe zeru");

            byte[,] zbior = new byte[Szerokosc, Wysokosc];

            Parallel.For( //x,y - wspolrzedne pikseli //for (int x = 0; x < bufor.Width; x++) 
                0, Szerokosc,
                (int x) =>
                {
                    for (int y = 0; y < Wysokosc; y++)
                    {
                        Complex c = 0, z = 0;

                        //Mandelbrot
                        if (zbiorMandelbrotaCzyJulii)
                        {
                            c = ClientToZespolone(x, y);
                            z = 0;
                        }
                        //Julia
                        if (!zbiorMandelbrotaCzyJulii)
                        {
                            c = new Complex(0.25, 0.55);
                            z = ClientToZespolone(x, y);
                        }

                        byte indeks = 0;
                        for (; indeks < N; indeks++)
                        {
                            z = z * z + c;
                            if (z.Magnitude >= 2) break;
                        }
                        if (indeks == N) zbior[x, y] = 0;
                        //bufor.SetPixel(x, y, Color.Black);
                        else
                            //bufor.SetPixel(x,y,Color.White);
                            //bufor.SetPixel(x, y, Color.FromArgb(255 - indeks, 255 - indeks, 255));                            
                            zbior[x, y] = (byte)(N - indeks);
                    }
                });

            return zbior;
        }

        public Task<byte[,]> ObliczAsync(bool zbiorMandelbrotaCzyJulii = true)
        {
            Task<byte[,]> t = new Task<byte[,]>(
                (object state) =>
                {
                    if(!(state is bool)) throw new ArgumentException("Argument powinien być wartością logiczną");
                    bool _zbiorMandelbrotaCzyJulii = (bool)state;
                    return Oblicz(_zbiorMandelbrotaCzyJulii);
                },
                zbiorMandelbrotaCzyJulii);
            t.Start();
            return t;
        }
    }
}
