﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Threading.Tasks;
using Windows.Storage;
using LightsOut.Data;
using LightsOut.Helpers;
using LightsOut.Interfaces;

namespace LightsOut
{
    public class ScoreManager : IScoreManager
    {
        #region Public properties
        /// <summary>
        /// Sorted list containing top scores
        /// </summary>
        public IReadOnlyList<PlayerScore> TopScores { get { return _topScores; } }
        #endregion

        #region Private fields
        private const int MaxNumberOfScores = 10;
        private List<PlayerScore> _topScores;
        #endregion

        #region Public methods
        /// <summary>
        /// Add score to the score list and write changes to file
        /// </summary>
        /// <param name="playerScore">New score to add to the list</param>
        public async Task AddScoreAsync(PlayerScore playerScore)
        {
            _topScores.Add(playerScore);
            _topScores.Sort();
            if (_topScores.Count > MaxNumberOfScores)
                _topScores.RemoveAt(MaxNumberOfScores);
            await WriteToFileAsync();
        }

        /// <summary>
        /// Write changes to file
        /// </summary>
        /// <param name="fileName">File name to save the changes to</param>
        /// <returns></returns>
        public async Task WriteToFileAsync(string fileName = "scores.xml")
        {
            var directory = ApplicationData.Current.RoamingFolder;
            var fileContent = Serializer.SerializeScoreList(_topScores);
            var file = await directory.CreateFileAsync(fileName, CreationCollisionOption.ReplaceExisting);
            await FileIO.WriteTextAsync(file, fileContent);
        }

        /// <summary>
        /// Read changes from file and reaplce current data
        /// </summary>
        /// <param name="fileName">File name to read new score data from</param>
        /// <returns></returns>
        public async Task ReadFromFileAsync(string fileName = "scores.xml")
        {
            try
            {
                var directory = ApplicationData.Current.RoamingFolder;
                var file = await directory.GetFileAsync(fileName);
                if (file == null)
                    return;
                var fileContent = await FileIO.ReadTextAsync(file);
                _topScores = Serializer.DeserializeScoreList(fileContent);
            }
            catch (FileNotFoundException)
            {
                _topScores = new List<PlayerScore>(MaxNumberOfScores + 1);
            }
        }
        #endregion
    }
}
