﻿using System;
using LightsOut.Data;
using LightsOut.Events;
using LightsOut.Helpers;
using LightsOut.Interfaces;

namespace LightsOut
{
    public class LightsOutGame : ILightsOutGame
    {
        #region Public properties
        public int RowCount { get; private set; }
        public int ColumnCount { get; private set; }
        public int LightsOnCount { get; private set; }
        public bool IsGameOver { get { return LightsOnCount == 0; } }
        public int MoveCount { get; private set; }

        public bool this[int row, int column]
        {
            get
            {
                return _gameGrid[row, column];
            }
            set
            {
                _gameGrid[row, column] = value;
                OnLightSwitched(new LightSwitchedEventArgs(row, column));
            }
        }
        #endregion

        #region Event handlers
        public event EventHandler<GameEndedEventArgs> GameEnded;
        public event EventHandler<LightSwitchedEventArgs> LightSwitched;
        #endregion

        #region Private fields
        private DateTime _gameStartTime;
        private readonly GameGrid _gameGrid;
        #endregion

        #region Constructor
        public LightsOutGame(int rowCount = 4, int columnCount = 4)
        {
            if (rowCount < 3 || columnCount < 3)
            {
                throw new ArgumentException("At least 3 RowCount and 3 ColumnCount are required!");
            }

            RowCount = rowCount;
            ColumnCount = columnCount;
            _gameGrid = new GameGrid(rowCount, columnCount);
        }
        #endregion

        #region Public Methods
        /// <summary>
        /// Carry out player's action at the specified row and column coordinates
        /// </summary>
        /// <param name="row">Row coordinate</param>
        /// <param name="column">Column coordinate</param>
        public virtual void PlayerAction(int row, int column)
        {
            if (IsGameOver)
            {
                return;
            }

            SwitchLight(row, column);
            SwitchLight(row - 1, column);
            SwitchLight(row + 1, column);
            SwitchLight(row, column - 1);
            SwitchLight(row, column + 1);
            MoveCount++;

            if (IsGameOver)
            {
                OnGameEnded(new GameEndedEventArgs(DateTime.UtcNow - _gameStartTime));
            }
        }

        /// <summary>
        /// Start a new game
        /// </summary>
        public virtual void NewGame()
        {
            MoveCount = 0;
            RandomizeGrid();
            _gameStartTime = DateTime.UtcNow;
        }
        #endregion

        #region Private methods
        /// <summary>
        /// Switch the light at the specified coordinates and call the LightSwitched event.
        /// </summary>
        /// <param name="row">Row coordinate</param>
        /// <param name="column">Column coordinate</param>
        private void SwitchLight(int row, int column)
        {
            // Get the actual coordinates if row or column exceeds the bounds (assuming periodic boundary conditions).
            row = BoundsCalculator.GetRow(row, RowCount);
            column = BoundsCalculator.GetColumn(column, ColumnCount);

            // Change the light count accordingly
            if (_gameGrid[row, column] == false)
            {
                LightsOnCount++;
            }
            else
            {
                LightsOnCount--;
            }

            // Switch the light sate and call the event
            _gameGrid[row, column] ^= true;
            OnLightSwitched(new LightSwitchedEventArgs(row, column));
        }

        /// <summary>
        /// Randomize light states across the grid
        /// </summary>
        private void RandomizeGrid()
        {
            var random = new Random(Guid.NewGuid().GetHashCode());
            LightsOnCount = 0;

            // Randomize the the light states
            for (int i = 0; i < RowCount; i++)
            {
                for (int j = 0; j < ColumnCount; j++)
                {
                    var isOn = random.Next(0, 2) == 1;
                    _gameGrid[i, j] = isOn;
                    if (isOn)
                        LightsOnCount++;
                    OnLightSwitched(new LightSwitchedEventArgs(i, j));
                }
            }

            // Make sure that at least 4 lights are on
            while (LightsOnCount < 4)
            {
                var row = random.Next(0, RowCount - 1);
                var column = random.Next(0, ColumnCount - 1);

                if (_gameGrid[row, column])
                    continue;

                _gameGrid[row, column] = true;
                LightsOnCount++;
                OnLightSwitched(new LightSwitchedEventArgs(row, column));
            }
        }
        #endregion

        #region Events
        protected virtual void OnLightSwitched(LightSwitchedEventArgs e)
        {
            var tmp = LightSwitched; //copy the delegate for thread safety
            if (tmp != null)
            {
                tmp(this, e);
            }
        }

        protected virtual void OnGameEnded(GameEndedEventArgs e)
        {
            var tmp = GameEnded; //copy the delegate for thread safety
            if (tmp != null)
            {
                tmp(this, e);
            }
        }
        #endregion
    }
}
