﻿/*
 * Created by:       Noah Coad, noah@coad.net, www.coad.net/noah
 * Started:          3/3/02
 * Version:          1.0.0
 * Credit:           Based off of Duane Odom's VB WinAmp control
 *                   http://www.winamp.com/nsdn/vault/WinAMP_VB.jhtml
 * Description:      Class for Controling WinAmp
 * Notes:            This code can be found at DevHood.com
 * 
 * 
     __  __                                           ______           __        
    |  |/  |.-----.-----.--.--.--.______.---.-.______|      |.-----.--|  |.-----.
    |     < |     |  _  |  |  |  |______|  _  |______|   ---||  _  |  _  ||  -__|
    |__|\__||__|__|_____|________|      |___._|      |______||_____|_____||_____|

 */
using System;
using System.IO;
using System.Text;
using System.ComponentModel;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace WinAmpControl
{
	/// <summary>
	/// Summary description for Class1.
	/// </summary>
	public class WinAmp
	{
		#region Constants
		private const int WM_COMMAND = 0x111;                // Used in SendMessage call
		private const int WM_USER = 0x400;                   // Used in SendMessage call
		private const string vbQuote = "\"";                 // Used in shelling to WinAMP

		#region Enumerations
		public enum PlayStatus {Playing, Stopped, Paused};
		
		public enum Commands
		{
			PrevTrack = 40044,
			NextTrack = 40048,
			Play = 40045,
			Pause = 40046,
			Stop = 40047,
			FadeOutStop = 40147,
			StopAfterTrack = 40157,
			FastForward = 40148,                       // 5 sec
			FastRewind = 40144,                        // 5 secs
			PlayListHome = 40154,
			DialogOpenFile = 40029,
			DialogOpenURL = 40155,
			DialogFileInfo = 40188,
			TimeDisplayElapsed = 40037,
			TimeDisplayRemaining = 40038,
			TogglePreferences = 40012,
			DialogVisualOptions = 40190,
			DialogVisualPluginOptions = 40191,
			StartVisualPlugin = 40192,
			ToggleAbout = 40041,
			ToggleAutoScroll = 40189,
			ToggleAlwaysOnTop = 40019,
			ToggleWindowShade = 40042,
			TogglePlayListWindowShade = 40266,
			ToggleDoublSize = 40165,
			ToggleEQ = 40036,
			TogglePlayList = 40040,
			ToggleMainWindow = 40258,
			ToggleMiniBrowser = 40298,
			ToggleEasyMode = 40186,
			VolumeUp = 40058,                          // Increase by 1%
			VolumeDown = 40059,                        // Decrease by 1%
			ToggleRepear = 40022,
			ToggleShuffle = 40023,
			DialogJumpToTime = 40193,
			DialogJumpToFile = 40194,
			DialogSkinSelector = 40219,
			DialogConfigureVisualPlugin = 40221,
			ReloadSkin = 40291,
			Close = 40001
		}

		public enum UserMessages
		{
			GetVersion = 0,
			ClearPlayList = 101,
			GetStatus = 104,
			GetTrackPosition = 105,
			GetTrackLength = 105,
			SeekToPosition = 106,
			SetVolume = 122,
			SetBallance = 123,
			GetEQData = 127,
			SetEQData = 128
		}
		#endregion

		// http://www.vbapi.com/ref/w/waveoutcaps.html
		[StructLayout(LayoutKind.Sequential, CharSet=CharSet.Ansi)]
		private struct WAVEOUTCAPS
		{
			public ushort wMid;
			public ushort wPid;
			public uint vDriverVersion;
			[ MarshalAs(UnmanagedType.ByValArray, SizeConst=32 )] 
			public char[] szPname;
			//public String szPname;                // Size 32
			public uint dwFormats;
			public ushort wChannels;              // The number of audio channels on the device. 1 means a mono device; 2 means a stereo device.
			public uint dwSupport;                // Zero or more of the following flags specifying which features the device supports:
		}

		private const int WAVE_FORMAT_1M08 = 0x1;            // Supports 11.025 kHz, 8-bit, mono playback.
		private const int WAVE_FORMAT_1M16 = 0x4;            // Supports 11.025 kHz, 16-bit, mono playback.
		private const int WAVE_FORMAT_1S08 = 0x2;            // Supports 11.025 kHz, 8-bit, stereo playback.
		private const int WAVE_FORMAT_1S16 = 0x8;            // Supports 11.025 kHz, 16-bit, stereo playback.
		private const int WAVE_FORMAT_2M08 = 0x10;           // Supports 22.05 kHz, 8-bit, mono playback.
		private const int WAVE_FORMAT_2M16 = 0x40;           // Supports 22.05 kHz, 16-bit, mono playback.
		private const int WAVE_FORMAT_2S08 = 0x20;           // Supports 22.05 kHz, 8-bit, stereo playback.
		private const int WAVE_FORMAT_2S16 = 0x80;           // Supports 22.05 kHz, 16-bit, stereo playback.
		private const int WAVE_FORMAT_4M08 = 0x100;          // Supports 44.1 kHz, 8-bit, mono playback.
		private const int WAVE_FORMAT_4M16 = 0x400;          // Supports 44.1 kHz, 16-bit, mono playback.
		private const int WAVE_FORMAT_4S08 = 0x200;          // Supports 44.1 kHz, 8-bit, stereo playback.
		private const int WAVE_FORMAT_4S16 = 0x800;          // Supports 44.1 kHz, 16-bit, stereo playback.

		private const int WAVECAPS_LRVOLUME = 0x8;           // Supports separate left and right channel volumes.
		private const int WAVECAPS_PITCH = 0x1;              // Supports pitch control.
		private const int WAVECAPS_PLAYBACKRATE = 0x2;       // Supports playback rate control.
		private const int WAVECAPS_SAMPLEACCURATE = 0x20;    // Supports returning of sample-accurate position information.
		private const int WAVECAPS_SYNC = 0x10;              // Supports synchronous playback -- i.e., it will block while playing buffered audio.
		private const int WAVECAPS_VOLUME = 0x4;             // Supports volume control.
		#endregion

		#region API Declarations
		// --------------------------------------
		//        Windows API Declarations
		// --------------------------------------
		[DllImport("user32")] private static extern ushort FindWindow(string lpClassName, string lpWindowName);
		[DllImport("user32")] private static extern ushort SendMessage(int hwnd, int wMsg, int wParam, int lParam);
		[DllImport("user32")] private static extern ushort GetWindowText(int hwnd, StringBuilder lpString, int nMaxCount);
		[DllImport("user32")] private static extern ushort GetWindowTextLength(int hwnd);
		[DllImport("winmm")] private static extern ushort waveOutGetVolume(int uDeviceID, out uint lpdwVolume);
		[DllImport("winmm")] private static extern ushort waveOutGetDevCaps(int uDeviceID, ref WAVEOUTCAPS lpCaps, int uSize);
		[DllImport("winmm")] private static extern ushort waveOutGetNumDevs();
		#endregion

		#region Private Local Variables
		private string m_WinAmpPath = "";
		private int m_hWndWinAMP;
		#endregion

		#region WinAmp Interfacting Public Methods & Properties

		/// <summary>
		/// Window Handler for the currently attached running WinAmp.
		/// Must be set to interact with WinAmp.  Can be changed to attach
		/// to other instances of WinAmp.
		/// </summary>
		public int hWndWinAMP
		{
			get {return m_hWndWinAMP;}
			set {m_hWndWinAMP = value;}
		}
		
		/// <summary>
		/// Obtains the name of the current song.
		/// </summary>
		/// <returns>Name of the current song.</returns>
		public string SongTitle
		{
			get
			{
				int textlen;                               // receives length of text of the window
				StringBuilder wintext;                     // receives the text of the window
	  
				// Obtain the title of the WinAmp dialog
				textlen = GetWindowTextLength(hWndWinAMP) + 1;
				wintext = new StringBuilder(textlen);
				GetWindowText(hWndWinAMP, wintext, textlen);
				
				// Extract the song title from the returned string
				string song = wintext.ToString();
				int start = song.IndexOf(". ") + 2;
				if (start == 1) return "";                 // No song is playing
				song = song.Substring(start, song.IndexOf(" - Winamp") - start);
				
				return song;
			}
		}

		
		/// <summary>
		/// Locates the handle for the WinAmp window and stores it in hWndWinAMP.
		/// </summary>
		/// <returns>True if WinAmp is successfully found.</returns>
		public bool FindWindow()
		{
			hWndWinAMP = FindWindow("Winamp v1.x", null);
			return (hWndWinAMP > 0);
		}
		

		/// <summary>
		/// Sets the EQ slider value for the slider specified by EQIndex
		/// </summary>
		/// <param name="EQIndex">Equalizer bar to adjust (0-9)</param>
		/// <param name="EQValue">Value of the eq setting (0 top - 63 bottom)</param>
		public void SetEQ(int EQIndex, int EQValue)
		{
			// Sets the EQ slider value for the slider specified by EQIndex
			// eq values will range from 0(top) to 63(bottom)
			
			// Settings do apply if the eq is turned on, but Eq Box does not update
			// it's slider bars or the wave display.  If a slider bar is changed
			// the wave display will reflect the changes, but the other sliders
			// will stay in place (not move to reflect the actual settings).
	    
			//range check the EQ Index
			if ((EQIndex < 0) | (EQIndex > 9)) throw new Exception("EQIndex out of range.");
	    
			// range check the new EQ value
			if ((EQValue < 0) | (EQValue > 63)) throw new Exception("EQValue out of range.");
	    
			// we have to query the eq line we want to change first
			SendMessage(hWndWinAMP, WM_USER, EQIndex, (int) UserMessages.GetEQData);
	    
			// now we send the new eq value to the selected eq line
			SendMessage(hWndWinAMP, WM_USER, EQValue, (int) UserMessages.SetEQData);
		}


		/// <summary>
		/// Gets or Sets the Pre-amp value
		/// </summary>
		/// <param name="PreAmpValue">Value of the pre-amp, 0 (top) to 63 (bottom).</param>
		public int PreAmp
		{
			// If the eq is turned on, the setting does take effect,
			// but is not reflected by the slider bar or wave display.
			set
			{
				//pre-amp values will range from 0(top) to 63(bottom)
				
				//range check the new pre-amp value
				if ((PreAmp < 0) | (PreAmp > 63)) throw new Exception("PreAmpValue is out of range.");
				
				// we have to query the pre-amp first
				int dummy = this.PreAmp;
				
				// now we send the new pre-amp value
				SendMessage(hWndWinAMP, WM_USER, PreAmp, (int) UserMessages.SetEQData);
			}
			get
			{
				// values will range from 0 (top) to 63 (bottom)
				return SendMessage(hWndWinAMP, WM_USER, 10, (int) UserMessages.GetEQData);
			}
		}


		/// <summary>
		/// Clears WinAMP's playlist
		/// </summary>
		public void ClearPlaylist()
		{
			SendMessage(hWndWinAMP, WM_USER, 0, (int) UserMessages.ClearPlayList);
		}


		/// <summary>
		/// Retrieves the status of WinAMP
		/// </summary>
		/// <returns>Playing, Paused, or Stopped</returns>
		public PlayStatus Status
		{
			get
			{
				int mode = SendMessage(hWndWinAMP, WM_USER, 0, (int) UserMessages.GetStatus);
				if (mode == 1) return PlayStatus.Playing;
				if (mode == 3) return PlayStatus.Paused;
				return PlayStatus.Stopped;
			}
		}

		
		/// <summary>
		/// Retrieves the position of the current track in seconds
		/// </summary>
		/// <returns></returns>
		public float TrackPosition
		{
			get
			{
				int ReturnPos;

				// ReturnPos will contain the current track pos in milliseconds
				// or -1 if no track is playing or an error occurs
				ReturnPos = SendMessage(hWndWinAMP, WM_USER, 0, (int) UserMessages.GetTrackPosition);

				if (ReturnPos != -1) return ReturnPos / 1000;   //convert ReturnPos to secs
				return -1;
			}
		}


		/// <summary>
		/// Retrieves the EQ slider value for the slider specified by EQIndex
		/// </summary>
		/// <param name="EQIndex">Index of the slider bar to retrieve.  From 0 to 9.</param>
		/// <returns>Returns the EQ value from 0 (top) to 63 (bottom).</returns>
		public int GetEQ(int EQIndex)
		{
			//range check the EQ Index (between 0 and 9, inclusive)
			if ((EQIndex < 0) | (EQIndex > 9)) throw new Exception("EQIndex is out of range.");
		  
			// return value will hold a value for the selected EQ slider
			// values will range from 0 (top) to 63 (bottom)
			return SendMessage(hWndWinAMP, WM_USER, EQIndex, (int) UserMessages.GetEQData);
		}


		/// <summary>
		/// Sets the volume slider (from 0 to 255)
		/// </summary>
		/// <param name="VolumeValue">Value from 0 to 255.</param>
		public int Volume
		{
			set
			{
				// volume values will range from 0 (bottom) to 255 (top)

				// range check the new volume
				if ((value < 0) | (value > 255)) throw new Exception("Volume out of range.");
				SendMessage(hWndWinAMP, WM_USER, value, (int) UserMessages.SetVolume);
			}

			get
			{
				//  Retrieve the current volume setting for waveform output device 0.  Note
				//  that we must first determine if separate volumes are returned or not, in order to know
				//  how to interpret the volume returned.  (We assume waveform output device #0 exists.)

				int retval;
				uint numvols, volume, lvolume, rvolume;

				WAVEOUTCAPS spkrcaps = new WAVEOUTCAPS();
				spkrcaps.szPname = new char[32];
				retval = waveOutGetDevCaps(-1, ref spkrcaps, Marshal.SizeOf(spkrcaps));
				if ((spkrcaps.dwSupport & WAVECAPS_LRVOLUME) == WAVECAPS_LRVOLUME) numvols = 2; else numvols = 1;
				retval = waveOutGetVolume(-1, out volume);

				if (numvols == 1) // if only one channel volume
					volume = volume & 0xFFFF;  // destroy irrelevant high-order word
				else
				{
					lvolume = volume & 0xFFFF;  // isolate left speaker volume
					rvolume = (volume & 0xFFFF0000) / 0x10000;  // isolate right speaker volume
					volume = lvolume;
				}

				return (int) (volume & 0xFF);
			}
		}


		/// <summary>
		/// Sets the Balance slider.
		/// </summary>
		/// <param name="BalanceValue">Value of the slider. -127 = All Left, 0 = Center, 127 = All Right</param>
		public int Balance
		{
			set
			{
				int ScaledBalanceValue;

				// we want the range of balance to be from -127 to 127, but the
				// real range is as follows

				// |LEFT                       CENTER                       RIGHT|
				// |128 ----------------------255(or 0)-----------------------127|
				// |_____________________________________________________________|
	    
				// range check the new balance
				if ((value < -127) | (value > 127)) throw new Exception("Ballance is out of range.");
			
				//here we do our shifting to correct the range of values
				if (value < 0) ScaledBalanceValue = 255 + value;
				else ScaledBalanceValue = value;

				SendMessage(hWndWinAMP, WM_USER, ScaledBalanceValue, (int) UserMessages.SetBallance);
			}
			get
			{
				return 0;
			}
		}


		/// <summary>
		/// Seeks to the specified position in the current track
		/// </summary>
		/// <param name="PositionInSec">Position in seconds</param>
		public void SeekToPosition(int PositionInSec)
		{
			// range check the new position
			if ((PositionInSec < 0) | (PositionInSec > this.TrackLength)) throw new Exception("PositionInSec is out of range.");

			SendMessage(hWndWinAMP, WM_USER, (int) (PositionInSec * 1000), (int) UserMessages.SeekToPosition);
		}


		/// <summary>
		/// The length of the current song in seconds.
		/// </summary>
		public int TrackLength
		{
			get
			{
				//Retrieves the length of the current track in secs'
				int length;
	    
				// length will contain the current track length in seconds
				// or -1 if no track is playing or an error occurs
				length = SendMessage(hWndWinAMP, WM_USER, 1, (int) UserMessages.GetTrackLength);

				return length;
			}
		}


		/// <summary>
		/// WinAmp's Version
		/// </summary>
		public string Version
		{
			get
			{
				int VersionNum; string ver, hex;

				VersionNum = SendMessage(hWndWinAMP, WM_USER, 0, (int) UserMessages.GetVersion);
				hex = VersionNum.ToString("X");
				
				if (hex.Length > 3)
				{
					hex = VersionNum.ToString("X");
					ver = hex.Substring(0, 1) + ".";
					ver += hex.Substring(1, 2) + ".";
					ver += hex.Substring(2, hex.Length - 2);
					return ver;
				}
				else return "Unknown";

			}
		}

		
		/// <summary>
		/// Causes WinAMP to open the file specified by filename
		/// </summary>
		/// <param name="filename">Name of the file to be opened.</param>
		/// <returns>Returns true if the file exists and WinAmp is located.</returns>
		public bool OpenFile(string filename)
		{
			return OpenFile(filename, System.Diagnostics.ProcessWindowStyle.Minimized);
		}

		/// <summary>
		/// Causes WinAMP to open the file specified by filename
		/// </summary>
		/// <param name="filename">Name of the file to be opened.</param>
		/// <param name="WindowStyle">Which Window Style WinAmp should be opened in, Normal, Minimized, Maximized, etc</param>
		/// <returns>Returns true if the file exists and WinAmp is located.</returns>
		public bool OpenFile(string filename, System.Diagnostics.ProcessWindowStyle WindowStyle)
		{
			if (System.IO.File.Exists(filename))
			{
				StartWinAmp("/ADD \"" + filename + "\"", WindowStyle);
				return true;
			}
			else return false;
		}

		
		/// <summary>
		/// Startup WinAmp, Waits for WinAmp to load and gets handle.
		/// </summary>
		public void Start()
		{
			Start(true);
		}

		/// <summary>
		/// Startup WinAmp
		/// </summary>
		/// <param name="wait">Wait for WinAmp to load, recommended.</param>
		public void Start(bool wait)
		{
			StartWinAmp("", ProcessWindowStyle.Normal);
			
			long TimeStamp = DateTime.Now.Ticks;
			if (wait) while (!FindWindow() & (DateTime.Now.Ticks - TimeStamp < TimeSpan.TicksPerSecond * 10)) System.Threading.Thread.Sleep(100);
		}
		

		/// <summary>
		/// Path and filename of WinAmp.exe, must be set for some opperations.
		/// </summary>
		public string WinAmpPath
		{
			get
			{
				if (!m_WinAmpPath.Equals("")) return m_WinAmpPath;

				// Look for WinAmp in the Registry
				string s = Microsoft.Win32.Registry.ClassesRoot.OpenSubKey("Applications").OpenSubKey("Winamp.exe").OpenSubKey("shell").OpenSubKey("open").OpenSubKey("command").GetValue("").ToString();
				if (s.Substring(s.Length - 5, 5).Equals(" \"%1\"")) s = s.Substring(0, s.Length - 5);
				s = s.Replace("\"", "");
				if (System.IO.File.Exists(s)) {m_WinAmpPath = s; return m_WinAmpPath;}
				
				// Look for WinAmp in the Default Directory
				string file = @"C:\Program Files\WinAmp\WinAmp.exe";
				if (System.IO.File.Exists(file)) {m_WinAmpPath = file; return m_WinAmpPath;}

				// Look for WinAmp in the Program Files Directory
				file = FindFile("WinAmp.exe", @"C:\Program Files");
				if (!file.Equals("")) {m_WinAmpPath = file; return file;}

				// Look for WinAmp anywhere on the C: Drive
				file = FindFile("WinAmp.exe");
				if (!file.Equals("")) {m_WinAmpPath = file; return file;}

				// Could not be found, throw execption.
				throw (new Exception("WinAmp Could Not Be Found"));
			}
			set
			{
				if (System.IO.File.Exists(value))
					m_WinAmpPath = value;
			}
		}


		/// <summary>
		/// Used to send any of the Command messages to WinAMP
		/// </summary>
		/// <param name="CommandMessage"></param>
		public void SendCommandMessage(Commands CommandMessage)
		{
			if (FindWindow()) SendMessage(hWndWinAMP, WM_COMMAND, (int) CommandMessage, 0);
		}


		#region Common Commands
		public void Play()
		{
			SendCommandMessage(Commands.Play);
		}
		public void PrevTrack()
		{
			SendCommandMessage(Commands.PrevTrack);
		}
		public void NextTrack()
		{
			SendCommandMessage(Commands.NextTrack);
		}

		public void Pause()
		{
			SendCommandMessage(Commands.Pause);
		}

		public void Stop()
		{
			SendCommandMessage(Commands.Stop);
		}
		#endregion
		
		#endregion

		#region Constructor(s)
		public WinAmp()
		{ FindWindow(); }
		#endregion

		#region Testing Entry Point
		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		[STAThread]
		
		#endregion

		#region Private Local Methods
		private static string FindFile(string filename)
		{
			return FindFile(filename, "C:\\");
		}

		private static string FindFile(string filename, string path)
		{
			string file;

			file = path + Path.DirectorySeparatorChar + filename;
			if (File.Exists(file)) return file;
			
			foreach (string dirs in Directory.GetDirectories(path))
			{
				file = FindFile(filename, dirs);
				if (!file.Equals("")) return file;
			}

			return "";
		}

		private void StartWinAmp(string args, System.Diagnostics.ProcessWindowStyle WindowStyle)
		{
			Process p = new Process();
			ProcessStartInfo psi = new ProcessStartInfo();
			psi.FileName = WinAmpPath;
			psi.Arguments = args;
			psi.WindowStyle = WindowStyle;
			p.StartInfo = psi;
			p.Start();
		}
		#endregion
	}
}
