#pragma once

#include <functional>

#include "rk1.hpp"
#include "rk2.hpp"
#include "rk4.hpp"

namespace rk_sse {
	template<uint rank, uint dim>
	struct StateVector_SSE {
		protected:
			static constexpr uint vsize = (rank * dim) + ((rank * dim) % 2);
			
			double * _data;
			double _h;
			std::function<void(double *, double *)> _step_function;

			virtual void step(double *, double *) = 0;
			
		public:
			static double maxError;

			StateVector_SSE(double h) {
				_h = h;
				_data = new double[vsize];
				_step_function = std::bind(&StateVector_SSE::step, this, std::placeholders::_1, std::placeholders::_2);
			}

			virtual ~StateVector_SSE() {
				delete[] _data;
			}

			inline double & operator[](unsigned int index) {
				return _data[index];
			}

			double getStep() {
				return _h;
			}

			void RK1SolverUpdate() {
				rk1<dim>(_data, _data, _step_function, _h);
			}

			void RK2SolverUpdate() {
				rk2<dim>(_data, _data, _step_function, _h);
			}

			void RK4SolverUpdate() {
				rk4<dim>(_data, _data, _step_function, _h);
			}

	};

	template<uint rank, uint dim>
	double StateVector_SSE<rank, dim>::maxError = 1e-5;
}
