using UnityEngine;
using System.Collections;
using System.Collections.Generic;

public class PlanetManager : MonoBehaviour
{
	private static PlanetManager _instance;		// singleton
	public static PlanetManager instance {
		get {
			if (_instance == null)		// Wykonywanie instancji programu tylko jeden raz
				_instance = GameObject.FindObjectOfType<PlanetManager> ();
			return _instance;
		}
	}

	private const string meshChildPrefix = "Mesh";
	private static string[] planetNames = {
		"Mercury",
		"Venus",
		"Earth",
		"Mars",
		"Jupiter",
		"Saturn",
		"Uranus",
		"Neptune"
	};
	private static string[] planetMaterials = {
		"Materials/MercuryMaterial",
		"Materials/VenusMaterial",
		"Materials/EarthMaterial", 
		"Materials/MarsMaterial",
		"Materials/JupiterMaterial",
		"Materials/SaturnMaterial",
		"Materials/UranusMaterial",
		"NeptuneMaterial"
	};
	private static string[] moonNames = { "Moon" };
	private static string[] moonParentNames = { "Earth" };
	private static string[] moonMaterials = { "Materials/MoonMaterial" };
	private GameObject rings;
	private float[][] planetParameters = new float[planetNames.Length][];
	private float[][] moonParameters = new float[moonNames.Length][];

	void Awake ()
	{
		_instance = this;

		float scaleRp = Scales.au2mu;

		
		const int numOfPlanetParams = 9;
		planetParameters [0] = new float[numOfPlanetParams] {
            0.205f, //ekscentryczno
			0.3075f * scaleRp, // r_pericenter
			0.241f, //orbital period
            0.3826f, //Promie rwnika wzgledem ziemskiego
            2.11f,//stopnie przechylenia planety
			-0.16f   , //rot period
			48.3f, //dugo geograficzna 
			0.05527f, //Masa podana w masach ziemskich
             0.3826f,
        };		//Mercury
		planetParameters [1] = new float[numOfPlanetParams] {
            0.007f,
			0.718f * scaleRp,
			0.615f,
            0.9488498f,
			177.3f,
			-0.67f  ,
			76.7f,
            0.81500f,
            0.9488498f,
        };		//Venus
		planetParameters [2] = new float[numOfPlanetParams] {
            0.017f,
			0.9833f * scaleRp,
			1f,
			1f,
			23.45f,
			-0.0027f,
			349f,
			1f,
            1f,
        };	        //Earth
		planetParameters [3] = new float[numOfPlanetParams] {
            0.094f,
			1.3814f * scaleRp,
			1.882f,
			0.53f,
			25.19f,
			-0.0029f,
			49.6f,
			0.1075f,
            0.53f,
        };		    //Mars
		planetParameters [4] = new float[numOfPlanetParams] {
                 0.049f,
			4.950f * scaleRp,
			11.86f ,
			11f,
			3.13f,
			-0.0011f ,
			101f,
			317.828f,
            2.7f
		};		//Jupiter
		planetParameters [5] = new float[numOfPlanetParams] {
            0.057f,
			9.021f * scaleRp,
			29.457f ,
			9.14f,
			26.73f,
			-0.0012f,
			114f,
			95.161f,
            2.4f
		};		//Saturn
		planetParameters [6] = new float[numOfPlanetParams] {
            0.046f,
			18.286f * scaleRp,
			84.016f ,
			4f,
			97.77f,
			0.0019f,
			74.2f,
			14.5357f,
            1.4f

		};	    //Uranus
		planetParameters [7] = new float[numOfPlanetParams] {
            0.011f,
			29.81f * scaleRp,
			164.791f,
			3.9f,
			28.32f,
			-0.0018f,
			131f,
			17.1478f,
            1.3f
		};		//Neptune

		const int numOfMoonParams = 8;
		moonParameters [0] = new float[numOfMoonParams] {
			0.055f,
			2.44f,
			0.074f,
			0.273f,
			6.69f,
			-29 / 365f,
			0.0f,
			0.012f
		};	

		for (int i = 0; i < planetNames.Length; i++)
			CreatePlanet (planetNames [i], Resources.Load (planetMaterials [i]) as Material, planetParameters [i]);

		for (int i = 0; i < moonNames.Length; i++)
			CreateMoon (moonNames [i], moonParentNames [i], Resources.Load (moonMaterials [i]) as Material, moonParameters [i]);

		rings = Instantiate (Resources.Load ("Prefabs/Rings") as GameObject) as GameObject;
		rings.transform.parent = Planet.planetList [5].transform.FindChild (meshChildPrefix + "Saturn");
		rings.transform.localScale = new Vector3 (5, 5, 5);
	}
    
	void CreatePlanet (string name, Material material, float[] par)
	{
		GameObject planet = Instantiate (Resources.Load ("Prefabs/PlanetHead") as GameObject) as GameObject;

		planet.name = name;
		planet.tag = "Planet";

		planet.layer = 9;

		AddRigidBody (ref planet, Scales.massScale * par [7]);

		planet.transform.FindChild ("Planet").name = meshChildPrefix + name;
		planet.transform.FindChild ("BB").name = "BB" + name;

		planet.transform.localScale = new Vector3 (par [8], par [8], par [8]);

		planet.GetComponent<PlanetaryOrbit> ().parametry_planet = par;
		planet.transform.parent = transform;

		AddSphereCollider (ref planet, Scales.gravityColliderMult * Scales.massScale * par [7]);
	}

	void CreateMoon (string name, string parentName, Material material, float[] par)
	{
		GameObject moon = Instantiate (Resources.Load ("Prefabs/PlanetHead") as GameObject) as GameObject;

		moon.name = name;
		moon.tag = "Moon";
		
		moon.transform.FindChild ("Planet").name = meshChildPrefix + name;
		moon.transform.FindChild ("BB").name = "BB" + name;
		
		moon.transform.localScale = new Vector3 (par [3], par [3], par [3]);
		
		moon.GetComponent<PlanetaryOrbit> ().parametry_planet = par;

		moon.transform.parent = GameObject.Find (parentName).transform;
	}

	void AddSphereCollider (ref GameObject planet, float radius)
	{
		planet.AddComponent<SphereCollider> ();
		planet.GetComponent<SphereCollider> ().center = Vector3.zero;
		planet.GetComponent<SphereCollider> ().radius = radius;
		planet.GetComponent<Collider> ().isTrigger = true;
	}

	void AddRigidBody (ref GameObject planet, float mass)
	{
		planet.AddComponent<Rigidbody> ();
		planet.GetComponent<Rigidbody> ().useGravity = false;
		planet.GetComponent<Rigidbody> ().angularDrag = 0f;
		planet.GetComponent<Rigidbody> ().mass = mass;
	}
}