﻿using Android.App;
using Android.Content;
using Android.Graphics;
using Android.Graphics.Drawables;
using Android.Graphics.Drawables.Shapes;
using Android.OS;
using Android.Preferences;
using Android.Support.V7.App;
using Android.Runtime;
using Android.Util;
using Android.Widget;
using Java.Lang;

namespace Kolory
{
    [Activity(Label = "@string/app_name", Theme = "@style/AppTheme", MainLauncher = true)]
    public class MainActivity : AppCompatActivity
    {
        private LinearLayout mainContainer;
        private TextView etykieta_panel;
        private SeekBar seekBarR, seekBarG, seekBarB;
        private Spinner spinner, colorModel;
        private bool ignoreSeekBarEvents;

        protected override void OnCreate(Bundle savedInstanceState)
        {
            base.OnCreate(savedInstanceState);
            Xamarin.Essentials.Platform.Init(this, savedInstanceState);
            // Set our view from the "main" layout resource
            SetContentView(Resource.Layout.activity_main);

            mainContainer = FindViewById<LinearLayout>(Resource.Id.main_container);
            mainContainer.Post(() => { SetSeekBarColors(0); });

            etykieta_panel = FindViewById<TextView>(Resource.Id.etykieta_panel);

            seekBarR = FindViewById<SeekBar>(Resource.Id.seekBarR);
            seekBarG = FindViewById<SeekBar>(Resource.Id.seekBarG);
            seekBarB = FindViewById<SeekBar>(Resource.Id.seekBarB);

            seekBarR.ProgressChanged += seekBar_ProgressChanged;
            seekBarG.ProgressChanged += seekBar_ProgressChanged;
            seekBarB.ProgressChanged += seekBar_ProgressChanged;

            //zmiana wygladu suwakow przeniesiona do setSeekBarColors

            //rozwijana lista
            spinner = FindViewById<Spinner>(Resource.Id.spinner);
            ArrayAdapter adapter2 = ArrayAdapter.CreateFromResource(this, Resource.Array.kolory, Android.Resource.Layout.SimpleSpinnerItem);
            adapter2.SetDropDownViewResource(Android.Resource.Layout.SimpleSpinnerDropDownItem);
            spinner.Adapter = adapter2;
            spinner.ItemSelected += spinner_ItemSelected;

            colorModel = FindViewById<Spinner>(Resource.Id.color_model_spinner);
            colorModel.SetSelection(0, false);
            colorModel.ItemSelected += ColorModel_ItemSelected;

            odtworzStan();
        }

        private void ColorModel_ItemSelected(object sender, AdapterView.ItemSelectedEventArgs e)
        {
            ignoreSeekBarEvents = true;
            FixSeekBarValues(e.Position);
            SetSeekBarColors(e.Position);
            ignoreSeekBarEvents = false;
        }

        private void SetSeekBarColors(int mode)
        {
            float promien = 16;
            float[] promienie = { promien, promien, promien, promien, promien, promien, promien, promien };
            RoundRectShape rrs = new RoundRectShape(promienie, null, null);
            switch (mode)
            {
                case 0:
                    ShapeDrawable sdR = new ShapeDrawable(rrs);
                    sdR.Paint.SetShader(new LinearGradient(0, 0, seekBarR.Width, 0, Color.Transparent, Color.Red, Shader.TileMode.Clamp));
                    seekBarR.ProgressDrawable = sdR;

                    ShapeDrawable sdG = new ShapeDrawable(rrs);
                    sdG.Paint.SetShader(new LinearGradient(0, 0, seekBarG.Width, 0, Color.Transparent, Color.Green, Shader.TileMode.Clamp));
                    seekBarG.ProgressDrawable = sdG;

                    ShapeDrawable sdB = new ShapeDrawable(rrs);
                    sdB.Paint.SetShader(new LinearGradient(0, 0, seekBarB.Width, 0, Color.Transparent, Color.Blue, Shader.TileMode.Clamp));
                    seekBarB.ProgressDrawable = sdB;

                    break;
                case 1:
                    ShapeDrawable sdH = new ShapeDrawable(rrs);
                    int[] rainbow = { Color.Red, Color.Yellow, Color.Green, Color.Cyan, Color.Blue, Color.Magenta, Color.Red };
                    Shader shader = new LinearGradient(0, 0, seekBarR.Width, 0, rainbow, null, Shader.TileMode.Clamp);
                    sdH.Paint.SetShader(shader);
                    seekBarR.ProgressDrawable = sdH;

                    UpdateSaturationSeekBarColor();
                    UpdateValueSeekBarColor();

                    break;
            }
        }

        private void UpdateSaturationSeekBarColor()
        {
            float promien = 16;
            float[] promienie = { promien, promien, promien, promien, promien, promien, promien, promien };
            RoundRectShape rrs = new RoundRectShape(promienie, null, null);

            Color saturation = Color.HSVToColor(new[] { seekBarR.Progress, 1f, 1f });

            ShapeDrawable sdS = new ShapeDrawable(rrs);
            sdS.Paint.SetShader(new LinearGradient(0, 0, seekBarB.Width, 0, Color.White, saturation, Shader.TileMode.Clamp));
            seekBarG.ProgressDrawable = sdS;

        }

        private void UpdateValueSeekBarColor()
        {
            float promien = 16;
            float[] promienie = { promien, promien, promien, promien, promien, promien, promien, promien };
            RoundRectShape rrs = new RoundRectShape(promienie, null, null);

            Color value = Color.HSVToColor(new[] { seekBarR.Progress, seekBarG.Progress / 100f, 1f });

            ShapeDrawable sdV = new ShapeDrawable(rrs);
            sdV.Paint.SetShader(new LinearGradient(0, 0, seekBarB.Width, 0, Color.Black, value, Shader.TileMode.Clamp));
            seekBarB.ProgressDrawable = sdV;
        }

        private void FixSeekBarValues(int mode)
        {
            switch (mode)
            {
                case 0:
                    var rgbFromHsv = Color.HSVToColor(new[] { seekBarR.Progress, seekBarG.Progress / 100f, seekBarB.Progress / 100f });
                    seekBarR.Max = seekBarG.Max = seekBarB.Max = 255;
                    seekBarR.Progress = rgbFromHsv.R;
                    seekBarG.Progress = rgbFromHsv.G;
                    seekBarB.Progress = rgbFromHsv.B;
                    break;
                case 1:
                    float[] hsvFromRgb = new float[3];
                    Color.RGBToHSV(seekBarR.Progress, seekBarG.Progress, seekBarB.Progress, hsvFromRgb);
                    seekBarR.Max = 359;
                    seekBarG.Max = seekBarB.Max = 100;
                    seekBarR.Progress = Math.Round(hsvFromRgb[0]);
                    seekBarG.Progress = Math.Round(hsvFromRgb[1] * 100f);
                    seekBarB.Progress = Math.Round(hsvFromRgb[2] * 100f);
                    break;
            }
        }

        protected override void OnPause()
        {
            base.OnPause();
            zapiszStan();
        }

        private string przestrzenKolorow = "przestrzen";
        private string skladowaR = "skladowaR";
        private string skladowaG = "skladowaG";
        private string skladowaB = "skladowaB";

        private void zapiszStan()
        {
            ISharedPreferencesEditor edytorUstawien = PreferenceManager.GetDefaultSharedPreferences(this).Edit();
            edytorUstawien.PutInt(przestrzenKolorow, colorModel.SelectedItemPosition);
            edytorUstawien.PutInt(skladowaR, seekBarR.Progress);
            edytorUstawien.PutInt(skladowaG, seekBarG.Progress);
            edytorUstawien.PutInt(skladowaB, seekBarB.Progress);
            edytorUstawien.Commit();
            Log.Info("Kolory", seekBarR.Progress + " " + seekBarG.Progress + " " + seekBarB.Progress);
        }

        private bool odtworzStan()
        {
            ISharedPreferences ustawienia = PreferenceManager.GetDefaultSharedPreferences(this);
            int przestrzen = ustawienia.GetInt(przestrzenKolorow, 0);
            if (przestrzen == 0)
            {
                seekBarR.Max = seekBarG.Max = seekBarB.Max = 255;
            }
            else
            {
                seekBarR.Max = 359;
                seekBarG.Max = seekBarB.Max = 100;
            }

            colorModel.SetSelection(przestrzen);

            seekBarR.Progress = ustawienia.GetInt(skladowaR, 0);
            seekBarG.Progress = ustawienia.GetInt(skladowaG, 0);
            seekBarB.Progress = ustawienia.GetInt(skladowaB, 0);

            return ustawienia.Contains(przestrzenKolorow) && ustawienia.Contains(skladowaR) && ustawienia.Contains(skladowaG) && ustawienia.Contains(skladowaB);
        }

        private void spinner_ItemSelected(object sender, AdapterView.ItemSelectedEventArgs e)
        {
            ignoreSeekBarEvents = true;
            switch (e.Position)
            {
                case 0:
                    break;
                case 1:
                    seekBarR.Progress = seekBarG.Progress = seekBarB.Progress = 0;
                    break;
                case 2:
                    seekBarR.Progress = seekBarG.Progress = seekBarB.Progress = 255;
                    break;
            }
            ignoreSeekBarEvents = false;
        }

        private void seekBar_ProgressChanged(object sender, SeekBar.ProgressChangedEventArgs e)
        {
            int r, g, b;
            float[] hsvComponents = new float[3];

            if (colorModel.SelectedItemPosition == 0)
            {
                r = seekBarR.Progress;
                g = seekBarG.Progress;
                b = seekBarB.Progress;

                Color.RGBToHSV(r, g, b, hsvComponents);
            }
            else
            {
                if (sender == seekBarR)
                {
                    UpdateSaturationSeekBarColor();
                    UpdateValueSeekBarColor();
                }
                else if (sender == seekBarG)
                {
                    UpdateValueSeekBarColor();
                }

                hsvComponents = new[] { seekBarR.Progress, seekBarG.Progress / 100f, seekBarB.Progress / 100f };
                Color color = Color.HSVToColor(hsvComponents);

                r = color.R;
                g = color.G;
                b = color.B;
            }

            etykieta_panel.SetBackgroundColor(Color.Argb(255, r, g, b));

            string rgb = "#" + r.ToString("X2").ToUpper() + g.ToString("X2").ToUpper() + b.ToString("X2").ToUpper();
            string hsv = Math.Round(hsvComponents[0]).ToString() + " " +
                         (hsvComponents[1]).ToString("0.##") + " " +
                         (hsvComponents[2]).ToString("0.##");

            etykieta_panel.Text = GetString(Resource.String.etykieta) + ":\nRGB: " + rgb + "\nHSV: " + hsv;

            int jasnosc = (int)(0.2126 * r + 0.7152 * g + 0.0722 * b);
            etykieta_panel.SetTextColor(jasnosc > 127 ? Color.Black : Color.White);

            if (!ignoreSeekBarEvents)
                spinner.SetSelection(0);
        }

        public override void OnRequestPermissionsResult(int requestCode, string[] permissions, [GeneratedEnum] Android.Content.PM.Permission[] grantResults)
        {
            Xamarin.Essentials.Platform.OnRequestPermissionsResult(requestCode, permissions, grantResults);

            base.OnRequestPermissionsResult(requestCode, permissions, grantResults);
        }
    }
}