﻿using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Shapes;

namespace GazeBreakoutWPF
{
    using BreakoutModel;

    /// <summary>
    /// Logika interakcji dla klasy MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {
        private GameSettings settings = null;
        private Model model;
        private System.Windows.Threading.DispatcherTimer dispatcherTimer;
        private Cursor originalCursor;

        public MainWindow()
        {
            settings = GameSettings.Default;

            InitializeComponent();

            //ominięcie babola Windows
            double width = SystemParameters.WorkArea.Width + SystemParameters.FixedFrameVerticalBorderWidth;
            double height = SystemParameters.WorkArea.Height + SystemParameters.FixedFrameHorizontalBorderHeight;
            this.Width = (int)width;
            this.Height = (int)height;

            GameLab.Geometry.Size canvasSize = new GameLab.Geometry.Size(800, 600);
            settings.InitialBallPositionX = canvasSize.Width / 2;
            settings.InitialBallPositionY = canvasSize.Height - 10 * settings.InitialPaddleSizeHeight;
            settings.InitialPaddlePositionX = canvasSize.Width / 2;
            settings.InitialPaddlePositionY = canvasSize.Height - 4 * settings.InitialPaddleSizeHeight;

            model = new Model(settings, canvasSize);
            model.PaddleMoved += model_PaddleMoved;
            model.BallMoved += model_BallMoved;
            model.BrickStateChanged += model_BrickStateChanged;
            model.LevelChanged += model_LevelChanged;

            createOrAdjustBricks();
            paddle.Width = model.Paddle.Width;
            paddle.Height = model.Paddle.Height;
            adjustPaddlePosition();
            ball.Width = model.Ball.Width;
            ball.Height = model.Ball.Height;
            adjustBallPosition();

            dispatcherTimer = new System.Windows.Threading.DispatcherTimer();
            dispatcherTimer.Tick += dispatcherTimer_Tick;
            dispatcherTimer.Interval = new TimeSpan(0, 0, 0, 0, settings.TimerIntervalMs);
            dispatcherTimer.Start();

            originalCursor = Cursor;
            Cursor = Cursors.None;
        }        

        private void model_PaddleMoved(object sender, EventArgs e)
        {
            adjustPaddlePosition();
        }

        private void model_BallMoved(object sender, EventArgs e)
        {
            adjustBallPosition();
        }

        private void model_BrickStateChanged(object sender, Model.BrickStateEventArgs e)
        {
            switch (e.NewState)
            {
                case Brick.BrickState.Normal:
                    break;
                case Brick.BrickState.Crushed:
                    bricks[e.Ix, e.Iy].Stroke = bricks[e.Ix, e.Iy].Fill;
                    break;
                case Brick.BrickState.Removed:
                    bricks[e.Ix, e.Iy].Stroke = Brushes.Transparent;
                    bricks[e.Ix, e.Iy].Fill = Brushes.Transparent;
                    break;
            }
        }

        private void model_LevelChanged(object sender, int e)
        {
            dispatcherTimer.Stop();
            messages.Text = "Następny poziom";
            messages.Visibility = Visibility.Visible;
            ball.Visibility = Visibility.Hidden;

            System.Windows.Threading.DispatcherTimer pauseTimer = new System.Windows.Threading.DispatcherTimer();
            pauseTimer.Tick += 
                (object _sender, EventArgs _e) =>
                {
                    pauseTimer.Stop();
                    messages.Visibility = Visibility.Hidden;
                    ball.Visibility = Visibility.Visible;
                    dispatcherTimer.Start();
                    createOrAdjustBricks(true);
                    adjustPaddlePosition();
                    adjustBallPosition();
                };
            pauseTimer.Interval = new TimeSpan(0, 0, 0, 2, 0); //2s
            pauseTimer.Start();          
        }

        private long timeMs;
        private bool tickExecuting = false;

        private void dispatcherTimer_Tick(object sender, EventArgs e)
        {
            if (tickExecuting) return;
            tickExecuting = true;

            timeMs += settings.TimerIntervalMs;

            if (settings.GameControlMode == GameSettings._GameControlMode.Keyboard)
            {
                if (KeyboardHelper.IsKeyDown(Key.A) || KeyboardHelper.IsKeyDown(Key.Left)) model.MovePaddleHorizontaly(Model.PaddleMoveDirection.Left);
                if (KeyboardHelper.IsKeyDown(Key.D) || KeyboardHelper.IsKeyDown(Key.Right)) model.MovePaddleHorizontaly(Model.PaddleMoveDirection.Right);
            }

            model.Update(timeMs); //czy nie przenieść timera do modelu i działać na zdarzeniach?
            text.Text = "Poziom: " + model.Level.ToString() + ", Punkty: " + model.Points.ToString();

            tickExecuting = false;
        }

        //view
        private Rectangle[,] bricks = null;

        private Brush getBrushForRow(int iy)
        {
            switch(iy)
            {
                case 0: return Brushes.Red;
                case 1: return Brushes.Orange;
                case 2: return Brushes.Yellow;
                case 3: return Brushes.Green;
                case 4: return Brushes.Cyan;
                case 5: return Brushes.Indigo;
                case 6: return Brushes.Violet;
                default: return Brushes.White;
            }
        }

        private void createOrAdjustBricks(bool reset = false)
        {            
            int nx = model.Bricks.GetLength(0);
            int ny = model.Bricks.GetLength(1);
            if (reset) bricks = null;
            if (bricks == null) bricks = new Rectangle[nx, ny];
            for (int iy = 0; iy < ny; ++iy)
            {
                for (int ix = 0; ix < nx; ++ix)
                {
                    GameLab.Geometry.Rectangle brickShape = model.Bricks[ix, iy].Rectangle;
                    Rectangle r;
                    if (bricks[ix, iy] == null)
                    {
                        r = new Rectangle();
                        r.Fill = getBrushForRow(iy);
                        r.Stroke = Brushes.White;
                        r.StrokeThickness = 3;
                        bricks[ix, iy] = r;
                        canvas.Children.Add(r);
                    }
                    else r = bricks[ix, iy];

                    r.Width = brickShape.Width;
                    r.Height = brickShape.Height;
                    Canvas.SetLeft(r, brickShape.Left);
                    Canvas.SetTop(r, brickShape.Top);  
                }
            }
        }

        private void adjustPaddlePosition()
        {
            Canvas.SetLeft(paddle, model.Paddle.Left);
            Canvas.SetTop(paddle, model.Paddle.Top);            
        }

        private void adjustBallPosition()
        {
            Canvas.SetLeft(ball, model.Ball.Left);
            Canvas.SetTop(ball, model.Ball.Top);
        }

        private void Window_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            double width = SystemParameters.WorkArea.Width + SystemParameters.FixedFrameVerticalBorderWidth;
            double height = SystemParameters.WorkArea.Height + SystemParameters.FixedFrameHorizontalBorderHeight;
            this.Width = (int)width;
            this.Height = (int)height;

            GameLab.Geometry.Size canvasSize = new GameLab.Geometry.Size((int)canvas.ActualWidth, (int)canvas.ActualHeight);
            settings.InitialBallPositionX = canvasSize.Width / 4;
            settings.InitialBallPositionY = canvasSize.Height - 10 * settings.InitialPaddleSizeHeight;
            settings.InitialPaddlePositionX = canvasSize.Width / 2;
            settings.InitialPaddlePositionY = canvasSize.Height - 4 * settings.InitialPaddleSizeHeight;            

            model.AdjustToSize(canvasSize);
            createOrAdjustBricks();
            adjustPaddlePosition();
            adjustBallPosition();
        }

        private void Window_MouseMove(object sender, MouseEventArgs e)
        {
            if (settings.GameControlMode == GameSettings._GameControlMode.Mouse)
            {
                int x = (int)e.GetPosition(this).X - model.Paddle.Width / 2;
                model.SetPaddlePosition(x);
            }
        }

        private void Window_Closed(object sender, EventArgs e)
        {
            this.Cursor = originalCursor;            
        }
    }
}
