﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace BreakoutModel
{
    using GameLab.Geometry;
    using static GameLab.Geometry.ComputationalGeometry;

    //tu nie ma odwraca (paletka u góry, dopiero na poziomie wyświetlania)
    public class Model
    {
        public int Points { get; private set; } = 0;
        public int Level { get; private set; } = 1;

        private Point ballPosition; //lewy górny róg
        private Size ballSize;

        private Point paddlePosition;
        private Size paddleSize;

        private double paddleVelocityX;

        public Rectangle Ball
        {
            get
            {
                return new Rectangle(ballPosition, ballSize);
            }
        }

        public Rectangle Paddle
        {
            get
            {
                return new Rectangle(paddlePosition, paddleSize);
            }
        }

        public Brick[,] Bricks { get; private set; } = null;

        private Size canvasSize;
        private GameSettings settings;

        public Model(GameSettings settings, Size canvasSize)
        {
            this.settings = settings;
            this.canvasSize = canvasSize;

            createOrAdjustBricks();
            createOrAdjustBall();
            createOrAdjustPaddle();
        }

        private void createOrAdjustBricks(bool reset = false)
        {
            int nx = settings.NumberOfBricksInRow;
            int ny = settings.NumberOfBrickRows;

            int brickSpacingX = settings.BrickSpacingX;
            int brickSpacingY = settings.BrickSpacingY;
            int brickSizeWidth = (canvasSize.Width - 2 * settings.BrickMarginX - (nx - 1) * brickSpacingX) / nx;
            int brickSizeHeight = settings.BrickSizeHeight;

            int firstRawPositionY = canvasSize.Height / 2 - ny * brickSizeHeight - (ny - 1) * brickSpacingY;

            if (reset) Bricks = null;
            if (Bricks == null) Bricks = new Brick[nx, ny];
            for (int iy = 0; iy < ny; ++iy)
            {
                for (int ix = 0; ix < nx; ++ix)
                {
                    int x = settings.BrickMarginX + ix * (brickSizeWidth + brickSpacingX);
                    int y = firstRawPositionY + iy * (brickSizeHeight + brickSpacingY);
                    if (Bricks[ix, iy] == null)
                    {
                        Brick brick = new Brick();
                        //brick.Rectangle = new Rectangle(x, y, brickSizeWidth, brickSizeHeight);
                        brick.State = Brick.BrickState.Normal;
                        Bricks[ix, iy] = brick;
                    }
                    Bricks[ix, iy].Rectangle = new Rectangle(x, y, brickSizeWidth, brickSizeHeight);
                }
            }
        }

        private void createOrAdjustBall()
        {
            this.ballSize = new Size(settings.InitialBallSizeWidth, settings.InitialBallSizeHeight);
            this.ballPosition = new Point(settings.InitialBallPositionX, settings.InitialBallPositionY);
            this.ballVelocityX = settings.InitialBallVelocityX;
            this.ballVelocityY = settings.InitialBallVelocityY;
        }

        private void createOrAdjustPaddle()
        {
            this.paddleSize = new Size(settings.InitialPaddleSizeWidth, settings.InitialPaddleSizeHeight);
            this.paddlePosition = new Point(settings.InitialPaddlePositionX, settings.InitialPaddlePositionY);
        }

        public void AdjustToSize(Size newScreenSize)
        {
            canvasSize = newScreenSize;
            createOrAdjustBricks();
            createOrAdjustBall();
            createOrAdjustPaddle();
        }

        public event EventHandler PaddleMoved;

        private void onPaddleMoved()
        {
            if (PaddleMoved != null) PaddleMoved(this, EventArgs.Empty);
        }

        public enum PaddleMoveDirection { Left, Right }

        public void MovePaddleHorizontaly(PaddleMoveDirection paddleMoveDirection)
        {
            switch (paddleMoveDirection)
            {
                case PaddleMoveDirection.Left:
                    paddlePosition.X -= settings.PaddleKeyboardMoveStep;
                    limitPaddlePositionX();
                    onPaddleMoved();
                    break;
                case PaddleMoveDirection.Right:
                    paddlePosition.X += settings.PaddleKeyboardMoveStep;
                    limitPaddlePositionX();
                    onPaddleMoved();
                    break;
            }
        }

        private int lastMousePositionX;

        private void setPaddleConstantVelocity(int x)
        {
            if (x < paddlePosition.X) paddleVelocityX = -settings.PaddleConstantVelocity;
            if (x > paddlePosition.X) paddleVelocityX = settings.PaddleConstantVelocity;
            if (Math.Abs(x - paddlePosition.X) < paddleSize.Width / 10) paddleVelocityX = 0;
            //else onPaddleMoved(); //jest w Update, niepotrzebne
        }

        private void setPaddleProportionalVelocity(int x)
        {
            if (Math.Abs(x - paddlePosition.X) < paddleSize.Width / 10) paddleVelocityX = 0;
            else paddleVelocityX = settings.PaddleProportionalVelocityCoefficient * (x - paddlePosition.X);
        }

        public void SetPaddlePosition(int x)
        {
            lastMousePositionX = x;
            switch (settings.PaddleControlMode)
            {
                case GameSettings._PaddleControlMode.Direct:
                    paddlePosition.X = x;
                    limitPaddlePositionX();
                    onPaddleMoved();
                    break;
                case GameSettings._PaddleControlMode.ConstantVelocity:
                    setPaddleConstantVelocity(x);
                    break;
                case GameSettings._PaddleControlMode.ProportionalVelocity:
                    setPaddleProportionalVelocity(x);
                    break;
            }
        }

        long prevTimeMs = 0;

        private void limitPaddlePositionX()
        {
            if (paddlePosition.X < 0) paddlePosition.X = 0;
            if (paddlePosition.X + paddleSize.Width > canvasSize.Width) paddlePosition.X = canvasSize.Width - paddleSize.Width;
        }

        private void calculateNextPaddlePositionEuler(double dt)
        {
            paddlePosition.X += (int)(paddleVelocityX * dt); //Euler
            limitPaddlePositionX();
        }

        public void Update(long timeMs)
        {
            double dt = timeMs - prevTimeMs;

            switch (settings.PaddleControlMode)
            {
                case GameSettings._PaddleControlMode.ConstantVelocity:
                    if (paddleVelocityX != 0)
                    {
                        calculateNextPaddlePositionEuler(dt);
                        setPaddleConstantVelocity(lastMousePositionX);
                        onPaddleMoved();
                    }
                    break;
                case GameSettings._PaddleControlMode.ProportionalVelocity:
                    if (paddleVelocityX != 0)
                    {
                        calculateNextPaddlePositionEuler(dt);
                        setPaddleProportionalVelocity(lastMousePositionX);
                        onPaddleMoved();
                    }
                    break;
            }

            //wpierw paleta, a dopiero potem piłka z ewentualną reakcją na ruchomą paletkę
            calculateNextBallPositionEuler(dt);

            if (NumberOfRemainingBricks == 0) nextLevel();

            prevTimeMs = timeMs;
        }

        private double ballVelocityX, ballVelocityY;
        private Point prevBallPosition;

        private void calculateNextBallPositionEuler(double dt)
        {
            ballPosition.X += (int)(ballVelocityX * dt); //Euler            
            ballPosition.Y += (int)(ballVelocityY * dt);
            ballEdgesBounces();
            ballPaddleBounces();
            ballBricksBounces();            
            onBallMoved();
            prevBallPosition = ballPosition;
        }

        public int NumberOfRemainingBricks
        {
            get
            {
                //pewnie optymalniej byłoby zmniejszać liczbę cegieł przy uderzeniach, ale przy tych ilościach to nie ma wpływu na wydajność
                int n = 0;
                foreach (Brick brick in Bricks)
                    if (brick.State != Brick.BrickState.Removed) n++;
                return n;
            }
        }

        public int NumberOfRemovedBricks
        {
            get
            {
                //pewnie optymalniej byłoby zmniejszać liczbę cegieł przy uderzeniach, ale przy tych ilościach to nie ma wpływu na wydajność
                int n = 0;
                foreach (Brick brick in Bricks)
                    if (brick.State == Brick.BrickState.Removed) n++;
                return n;
            }
        }

        private void ballEdgesBounces()
        {
            if (ballPosition.X < 0)
            {
                ballVelocityX = -ballVelocityX;
                ballPosition.X = 0; //wyrównanie, żeby było widać moment zderzenia
            }
            if (ballPosition.X + ballSize.Width > canvasSize.Width)
            {
                ballVelocityX = -ballVelocityX;
                ballPosition.X = canvasSize.Width - ballSize.Width;
            }
            if (ballPosition.Y < 0)
            {
                ballVelocityY = -ballVelocityY;
                ballPosition.Y = 0;
            }
            //tu wykrycie przegranej - opcja
            if (ballPosition.Y + ballSize.Height > canvasSize.Height)
            {
                ballVelocityY = -ballVelocityY;
                ballPosition.Y = canvasSize.Height - ballSize.Height;
            }
        }

        private bool isBallRectangleCollision(Point rectanglePosition, Size rectangleSize)
        {
            return
                ballPosition.Y + ballSize.Height > rectanglePosition.Y &&
                ballPosition.Y < rectanglePosition.Y + rectangleSize.Height &&
                ballPosition.X + ballSize.Width > rectanglePosition.X &&
                ballPosition.X < rectanglePosition.X + rectangleSize.Width;
        }

        //kulka jest rysowana jako okrągła, ale odbija się jakby była kwadratowa
        private bool ballRectangleBounces(Point rectanglePosition, Size rectangleSize, bool overridePhysics = false)
        {
            if (isBallRectangleCollision(rectanglePosition, rectangleSize))
            {
                Point? pi_left, pi_top, pi_right, pi_bottom;
                bool bi_left = IsLineSegmentsIntersecting(new Point(prevBallPosition.X + ballSize.Width, prevBallPosition.Y), new Point(ballPosition.X + ballSize.Width, ballPosition.Y), rectanglePosition, new Point(rectanglePosition.X, rectanglePosition.Y + rectangleSize.Height), out pi_left);
                if(!bi_left) bi_left = IsLineSegmentsIntersecting(new Point(prevBallPosition.X + ballSize.Width, prevBallPosition.Y + ballSize.Height), new Point(ballPosition.X + ballSize.Width, ballPosition.Y + ballSize.Height), rectanglePosition, new Point(rectanglePosition.X, rectanglePosition.Y + rectangleSize.Height), out pi_left);
                bool bi_top = IsLineSegmentsIntersecting(new Point(prevBallPosition.X, prevBallPosition.Y + ballSize.Height), new Point(ballPosition.X, ballPosition.Y + ballSize.Height), rectanglePosition, new Point(rectanglePosition.X + rectangleSize.Width, rectanglePosition.Y), out pi_top);
                if(!bi_top) bi_top = IsLineSegmentsIntersecting(new Point(prevBallPosition.X + ballSize.Width, prevBallPosition.Y + ballSize.Height), new Point(ballPosition.X + ballSize.Width, ballPosition.Y + ballSize.Height), rectanglePosition, new Point(rectanglePosition.X + rectangleSize.Width, rectanglePosition.Y), out pi_top);
                bool bi_right = IsLineSegmentsIntersecting(prevBallPosition, ballPosition, new Point(rectanglePosition.X + rectangleSize.Width, rectanglePosition.Y), new Point(rectanglePosition.X + rectangleSize.Width, rectanglePosition.Y + rectangleSize.Height), out pi_right);
                if(!bi_right) bi_right = IsLineSegmentsIntersecting(new Point(prevBallPosition.X, prevBallPosition.Y + ballSize.Height), new Point(ballPosition.X, ballPosition.Y + ballSize.Height), new Point(rectanglePosition.X + rectangleSize.Width, rectanglePosition.Y), new Point(rectanglePosition.X + rectangleSize.Width, rectanglePosition.Y + rectangleSize.Height), out pi_right);
                bool bi_bottom = IsLineSegmentsIntersecting(prevBallPosition, ballPosition, new Point(rectanglePosition.X, rectanglePosition.Y + rectangleSize.Height), new Point(rectanglePosition.X + rectangleSize.Width, rectanglePosition.Y + rectangleSize.Height), out pi_bottom);
                if(!bi_bottom) bi_bottom = IsLineSegmentsIntersecting(new Point(prevBallPosition.X + ballSize.Width, prevBallPosition.Y), new Point(ballPosition.X + ballSize.Width, ballPosition.Y), new Point(rectanglePosition.X, rectanglePosition.Y + rectangleSize.Height), new Point(rectanglePosition.X + rectangleSize.Width, rectanglePosition.Y + rectangleSize.Height), out pi_bottom);
                //szukam najbliższego punktu przecięcia
                double? di_left = null, di_top = null, di_right = null, di_bottom = null;

                List<double> di = new List<double>();
                if (pi_left.HasValue) { di_left = Distance(new Point(prevBallPosition.X + ballSize.Width, prevBallPosition.Y), pi_left.Value); di.Add(di_left.Value); }
                if (pi_top.HasValue) { di_top = Distance(new Point(prevBallPosition.X, prevBallPosition.Y + ballSize.Height), pi_top.Value); di.Add(di_top.Value); }
                if (pi_right.HasValue) { di_right = Distance(prevBallPosition, pi_right.Value); di.Add(di_right.Value); }
                if (pi_bottom.HasValue) { di_bottom = Distance(prevBallPosition, pi_bottom.Value); di.Add(di_bottom.Value); }

                if (di.Count > 0)
                {
                    double di_min = di.Min();

                    if (di_left.HasValue && di_min == di_left)
                    {
                        ballVelocityX = -ballVelocityX;
                        ballPosition.X = rectanglePosition.X - ballSize.Width - 1;
                    }
                    if (di_right.HasValue && di_min == di_right)
                    {
                        ballVelocityX = -ballVelocityX;
                        ballPosition.X = rectanglePosition.X + rectangleSize.Width + 1;
                    }
                    if (di_top.HasValue && di_min == di_top)
                    {
                        ballVelocityY = -ballVelocityY;
                        if(overridePhysics && pi_top.HasValue)
                        {
                            double c = 2.0 * (ballPosition.X + ballSize.Width / 2 - rectanglePosition.X) / rectangleSize.Width - 1.0;
                            ballVelocityX += c * 1.0;
                            if (ballVelocityX < -settings.MaximalBallVelocityX) ballVelocityX = -settings.MaximalBallVelocityX;
                            if (ballVelocityX > settings.MaximalBallVelocityX) ballVelocityX = settings.MaximalBallVelocityX;
                        }
                        ballPosition.Y = rectanglePosition.Y - ballSize.Height - 1;
                    }
                    if (di_bottom.HasValue && di_min == di_bottom)
                    {
                        ballVelocityY = -ballVelocityY;
                        ballPosition.Y = rectanglePosition.Y + rectangleSize.Height + 1;
                    }

                    return true;
                }
            }
            return false;
        }

        private bool ballRectangleBounces(Rectangle rectangle)
        {
            return ballRectangleBounces(rectangle.Position, rectangle.Size);
        }

        private bool ballPaddleBounces()
        {
            return ballRectangleBounces(paddlePosition, paddleSize, settings.PaddleOverridePhysics);
        }

        public class BrickStateEventArgs : EventArgs
        {
            public int Ix, Iy;
            public Brick.BrickState NewState;
        }

        public event EventHandler<BrickStateEventArgs> BrickStateChanged;

        private void onBrickStateChanged(int ix, int iy, Brick.BrickState newState)
        {
            if (BrickStateChanged != null)
                BrickStateChanged(
                    this,
                    new BrickStateEventArgs() { Ix = ix, Iy = iy, NewState = newState }
                );
        }

        private void ballBricksBounces()
        {
            int nx = settings.NumberOfBricksInRow;
            int ny = settings.NumberOfBrickRows;

            for (int iy = 0; iy < ny; ++iy)
            {
                for (int ix = 0; ix < nx; ++ix)
                {                    
                    if (Bricks[ix, iy].State != Brick.BrickState.Removed)
                    {
                        if (ballRectangleBounces(Bricks[ix, iy].Rectangle))
                        {
                            Points++;                            
                            if (settings.BricksCrushing) Bricks[ix, iy].State--;
                            else Bricks[ix, iy].State = Brick.BrickState.Removed;
                            onBrickStateChanged(ix, iy, Bricks[ix, iy].State);                            
                        }
                    }
                }
            }
        }

        public event EventHandler BallMoved;

        private void onBallMoved()
        {
            if (BallMoved != null) BallMoved(this, EventArgs.Empty);
        }

        public event EventHandler<int> LevelChanged;

        private void onLevelChanged()
        {
            if (LevelChanged != null) LevelChanged(this, settings.NumberOfBrickRows);
        }

        private void reset()
        {
            Bricks = null;
        }

        private void nextLevel()
        {
            Level++;
            Points += 10;

            settings.NumberOfBrickRows++;
            if(settings.NumberOfBrickRows >= settings.MaximalNumberOfBrickRows)
            {
                settings.NumberOfBrickRows = 1;
                settings.BricksCrushing = true;
            }

            createOrAdjustBricks(true);
            createOrAdjustBall();
            createOrAdjustPaddle();

            onLevelChanged();
        }
    }
}
