#ifndef MACIERZ_H
#define MACIERZ_H

#include <iostream>
#include <math.h>
using namespace std;

template <class T>
class Macierz
{
private:
    int wiersze;
    int kolumny;
    int rozmiar;
    
public:
    T* tablica;
    
    //Konstruktory/destruktory
    Macierz();
    Macierz(int wiersze, int kolumny, bool wyczysc=false);
    Macierz(const Macierz<T>&); //konstruktor kopiujacy
    Macierz(const T tablica[], const int, const int); //przypisz macierzy tablice o zadanych wymiarach
    ~Macierz();

    //Metody statyczne, w razie jak ktos by wolal ich w ten sposob uzywac
    static Macierz dodaj(const Macierz&, const Macierz&);
    static Macierz odejmij(const Macierz&, const Macierz&);
    static Macierz iloczyn(const Macierz&, const Macierz&);
    static void jednostkowa(Macierz&);
    static void jednostkowa(Macierz&, float, float, float, float);
    static void skalujaca(Macierz&, float, float, float);
    static void translacji(Macierz&, float, float, float);
    static void obrotuX(Macierz&, float);
    static void obrotuY(Macierz&, float);
    static void obrotuZ(Macierz&, float);
    
    //Metody niestatyczne
    void transponuj();
    void jednostkowa(); //zamien macierz na jednostkowa o x=1 y=1 z=1 w=1
    void jednostkowa(float, float, float, float); //zamien macierz na jednostkowa korzystajac z parametrow x, y, z i w
    void skalujaca(float, float, float); //zamien macierz na skalujaca wedlug x, y i z
    void translacji(float, float, float); //zamien macierz na macierz translacji wg x, y, i z
    void obrotuX(float); //zamien macierz na macierz obrotu wg X (left handed)
    void obrotuY(float); //jak wyzej, wg Y
    void obrotuZ(float); //jak wyzej, Z
    void wyczysc(); //wyzeruj macierz
    Macierz dodaj(const Macierz&);
    Macierz odejmij(const Macierz&);
    Macierz iloczyn(const Macierz&);

    //Operatory
    Macierz operator + (const Macierz&);
    Macierz operator - (const Macierz&);
    Macierz operator * (const Macierz&);
    void operator = (const Macierz&);
    T& operator [] (int);
    
    template <typename U>
    friend ostream& operator << (ostream& os, const Macierz<U>& m);
};


//----------------------Konstruktory/Destruktory----------------------
template <typename  T>
Macierz<T>::Macierz()
{
    wiersze = 0;
    kolumny = 0;
    rozmiar = 0;
    tablica = NULL;
}

template <typename  T>
Macierz<T>::Macierz(int wiersze, int kolumny, bool wyczysc)
{
    this->wiersze = wiersze;
    this->kolumny = kolumny;
    rozmiar = wiersze * kolumny;
    tablica = new T [rozmiar];
    
    if(wyczysc)
        this->wyczysc();
}

template <typename  T>
Macierz<T>::Macierz(const Macierz<T> &zrodlowa)
{
    wiersze = zrodlowa.wiersze;
    kolumny = zrodlowa.kolumny;
    rozmiar = zrodlowa.rozmiar;
    tablica = new T [rozmiar];
    for(int i=0; i<rozmiar; i++)
        tablica[i] = zrodlowa.tablica[i];
}

template <typename  T>
Macierz<T>::Macierz(const T tablica[], const int wiersze, const int kolumny)
{
    this->wiersze = wiersze;
    this->kolumny = kolumny;
    this->rozmiar = wiersze*kolumny;
    
    this->tablica = new T [rozmiar];
    for(int i=0; i<rozmiar; i++)
    {
        this->tablica[i] = tablica[i];
    }
}

template <typename  T>
Macierz<T>::~Macierz()
{
    delete tablica;
}


//----------------------Metody statyczne----------------------
template <typename  T>
Macierz<T> Macierz<T>::dodaj(const Macierz<T>& m1, const Macierz<T>& m2)
{
    if(m1.wiersze != m2.wiersze || m1.kolumny != m2.kolumny)
        throw;

    int rozmiar = m1.rozmiar;
    Macierz<T> suma = Macierz(m1.wiersze, m1.kolumny);
    
    for(int i=0; i<rozmiar; i++)
    {
        suma.tablica[i] = m1.tablica[i] + m2.tablica[i];
    }
    return suma;
}

template <typename  T>
Macierz<T> Macierz<T>::odejmij(const Macierz<T>& m1, const Macierz<T>& m2)
{
    if(m1.wiersze != m2.wiersze || m1.kolumny != m2.kolumny)
        throw;

    int rozmiar = m1.rozmiar;
    Macierz<T> roznica = Macierz<T>(m1.wiersze, m1.kolumny);
    
    for(int i=0; i<rozmiar; i++)
    {
        roznica.tablica[i] = m1.tablica[i] - m2.tablica[i];
    }
    return roznica;
}

template <typename  T>
Macierz<T> Macierz<T>::iloczyn(const Macierz<T>& m1, const Macierz<T>& m2)
{
    if(m1.wiersze != m2.kolumny)
        throw;
        
    Macierz wynik(m1.wiersze, m2.kolumny);
    for(int i=0; i<m1.wiersze; i++)
    {
        for(int j=0; j<m2.kolumny; j++)
        {
            wynik[(i*m2.kolumny)+j] = 0;
            for(int k=0; k<m2.wiersze; k++)
            {
                wynik[(i*m2.kolumny)+j] += m1.tablica[(i*m1.kolumny)+k] * m2.tablica[(k*m2.kolumny)+j];
            }
        }
    }
    return wynik;
}

template <typename  T>
void Macierz<T>::jednostkowa(Macierz<T>& m)
{
    if(m.wiersze != m.kolumny)
        throw;
    for(int i=0; i<m.wiersze; i++)
    {
        for(int j=0; j<m.wiersze; j++)
        {
            if(i==j)
                m.tablica[(i*m.wiersze)+j] = 1;
            else
                m.tablica[(i*m.wiersze)+j] = 0;
        }
    }
}

template <typename  T>
void Macierz<T>::jednostkowa(Macierz<T>& m, float x, float y, float z, float w)
{
    if(m.wiersze != m.kolumny || m.wiersze != 4)
        throw;
    for(int i=0; i<m.wiersze; i++)
    {
        for(int j=0; j<m.wiersze; j++)
        {
            if(i==j)
            {
                if(i==0)
                    m.tablica[(i*m.wiersze)+j] = x;
                else if(i==1)
                    m.tablica[(i*m.wiersze)+j] = y;
                else if(i==2)
                    m.tablica[(i*m.wiersze)+j] = z;
                else
                    m.tablica[(i*m.wiersze)+j] = w;
            }
            else
                m.tablica[(i*m.wiersze)+j] = 0;
        }
    }
}

template <typename  T>
void Macierz<T>::skalujaca(Macierz<T>& m, float x, float y, float z)
{
    if(m.wiersze != m.kolumny || m.wiersze != 4)
        throw;
    
    for(int i=0; i<m.wiersze; i++)
    {
        for(int j=0; j<m.wiersze; j++)
        {
            if(i==j)
            {
                if(i==0)
                    m.tablica[(i*m.wiersze)+j] = x;
                else if(i==1)
                    m.tablica[(i*m.wiersze)+j] = y;
                else if(i==2)
                    m.tablica[(i*m.wiersze)+j] = z;
                else
                    m.tablica[(i*m.wiersze)+j] = 1;
            }
            else
                m.tablica[(i*m.wiersze)+j] = 0;
        }
    }
}

template <typename  T>
void Macierz<T>::translacji(Macierz<T>& m, float x, float y, float z)
{
    if(m.wiersze != m.kolumny || m.wiersze != 4)
        throw; //dodac wyjatek
    for(int i=0; i<m.wiersze; i++)
    {
        for(int j=0; j<m.wiersze; j++)
        {
            if(i==j)
            {
                m.tablica[(i*m.wiersze)+j] = 1;
            }
            else
            {
                if(j == 3)
                {
                    if(i == 0)
                        m.tablica[(i*m.wiersze)+j] = x;
                    else if(i == 1)
                        m.tablica[(i*m.wiersze)+j] = y;
                    else if(i == 2)
                        m.tablica[(i*m.wiersze)+j] = z;
                }
                else
                    m.tablica[(i*m.wiersze)+j] = 0;
            }
        }
    }
}

template <typename  T>
void Macierz<T>::obrotuX(Macierz<T>& m, float phi)
{
    if(m.wiersze != m.kolumny || m.wiersze != 4)
        throw;
    for(int i=0; i<m.wiersze; i++)
    {
        for(int j=0; j<m.wiersze; j++)
        {
            if(i==j && (i==0 || i==3))
                m.tablica[(i*m.wiersze)+j] = 1;
            else if(i == 1)
            {
                if(j==1)
                    m.tablica[(i*m.wiersze)+j] = cos(phi);
                else if(j==2)
                    m.tablica[(i*m.wiersze)+j] = -sin(phi);
                else
                    m.tablica[(i*m.wiersze)+j] = 0;
            }
            else if(i==2)
            {
                if(j==1)
                    m.tablica[(i*m.wiersze)+j] = sin(phi);
                else if(j==2)
                    m.tablica[(i*m.wiersze)+j] = cos(phi);
                else
                    m.tablica[(i*m.wiersze)+j] = 0;
            }
            else
                m.tablica[(i*m.wiersze)+j] = 0;
        }
    }
}


template <typename  T>
void Macierz<T>::obrotuY(Macierz<T>& m, float theta)
{
    if(m.wiersze != m.kolumny || m.wiersze != 4)
        throw;
    for(int i=0; i<m.wiersze; i++)
    {
        for(int j=0; j<m.wiersze; j++)
        {
            if(i==j && (i==1 || i==3))
                m.tablica[(i*m.wiersze)+j] = 1;
            else if(i == 0)
            {
                if(j==0)
                    m.tablica[(i*m.wiersze)+j] = cos(theta);
                else if(j==2)
                    m.tablica[(i*m.wiersze)+j] = sin(theta);
                else
                    m.tablica[(i*m.wiersze)+j] = 0;
            }
            else if(i==2)
            {
                if(j==0)
                    m.tablica[(i*m.wiersze)+j] = -sin(theta);
                else if(j==2)
                    m.tablica[(i*m.wiersze)+j] = cos(theta);
                else
                    m.tablica[(i*m.wiersze)+j] = 0;
            }
            else
                m.tablica[(i*m.wiersze)+j] = 0;
        }
    }
}

template <typename  T>
void Macierz<T>::obrotuZ(Macierz<T>& m, float psi)
{
    if(m.wiersze != m.kolumny || m.wiersze != 4)
        throw;
    for(int i=0; i<m.wiersze; i++)
    {
        for(int j=0; j<m.wiersze; j++)
        {
            if(i==j && (i==2 || i==3))
                m.tablica[(i*m.wiersze)+j] = 1;
            else if(i == 0)
            {
                if(j==0)
                    m.tablica[(i*m.wiersze)+j] = cos(psi);
                else if(j==1)
                    m.tablica[(i*m.wiersze)+j] = -sin(psi);
                else
                    m.tablica[(i*m.wiersze)+j] = 0;
            }
            else if(i==1)
            {
                if(j==0)
                    m.tablica[(i*m.wiersze)+j] = sin(psi);
                else if(j==1)
                    m.tablica[(i*m.wiersze)+j] = cos(psi);
                else
                    m.tablica[(i*m.wiersze)+j] = 0;
            }
            else
                m.tablica[(i*m.wiersze)+j] = 0;
        }
    }
}

//----------------------Metody niestatyczne----------------------
template <typename  T>
void Macierz<T>::wyczysc()
{
    for(int i=0; i<rozmiar; i++)
        tablica[i] = 0;
}

template <typename  T>
void Macierz<T>::transponuj()
{   
    T* transposed = new T [rozmiar];
    for(int i=0; i<wiersze; i++)
    {
        for(int j=0; j<kolumny; j++)
        {
            transposed[(j*wiersze)+i] = tablica[(i*kolumny)+j];
        }
    }
    
    delete tablica;
    tablica = transposed;
    
    int temp = wiersze;
    wiersze = kolumny;
    kolumny = temp;
}

template <typename  T>
void Macierz<T>::jednostkowa()
{
    jednostkowa(*this);
}

template <typename  T>
void Macierz<T>::jednostkowa(float x, float y, float z, float w)
{
    jednostkowa(*this, x, y, z, w);
}

template <typename  T>
void Macierz<T>::skalujaca(float x, float y, float z)
{
    skalujaca(*this, x, y, z);
}

template <typename  T>
void Macierz<T>::translacji(float x, float y, float z)
{
    translacji(*this, x, y, z);
}

template <typename  T>
void Macierz<T>::obrotuX(float phi)
{
    obrotuX(*this, phi);
}

template <typename  T>
void Macierz<T>::obrotuY(float theta)
{
    obrotuY(*this, theta);
}

template <typename  T>
void Macierz<T>::obrotuZ(float psi)
{
    obrotuZ(*this, psi);
}

//----------------------Operatory----------------------
template <typename  T>
Macierz<T> Macierz<T>::operator +(const Macierz<T>& m2)
{
    return dodaj(*this, m2);
}

template <typename  T>
Macierz<T> Macierz<T>::operator -(const Macierz& m2)
{
    return odejmij(*this, m2);
}

template <typename  T>
Macierz<T> Macierz<T>::operator *(const Macierz<T>& m2)
{
    return iloczyn(*this, m2);
}

template <typename  T>
void Macierz<T>::operator =(const Macierz &m2)
{
    wiersze = m2.wiersze;
    kolumny = m2.kolumny;
    rozmiar = m2.rozmiar;
    
    tablica = new T [rozmiar];
    for(int i=0; i<rozmiar; i++)
    {
        tablica[i] = m2.tablica[i];
    }
}

template <typename  T>
T& Macierz<T>::operator [](int x)
{
    return tablica[x];
}

template <typename  T>
ostream& operator <<(ostream& os, const Macierz<T>& m)
{
    for(int i=0; i<m.rozmiar; i++)
    {
        os << m.tablica[i] << " ";
        if( ((i+1)%m.kolumny) == 0)
            os << endl;
    }
    return os;
}

#endif // MACIERZ_H
