#include "mygame.h"

#include "vertextshader.h"
#include "fragmentshader.h"
#include "geometryshader.h"

#include <sstream> // for std::stringstream
#include <iostream> // for std::cout
#include <fstream> // for std::fstream
#include <iomanip> // for std::setprecison

using namespace Gx;

static char logBuffer[1024];

#define BUFFER_OFFSET(bytes) ((GLubyte*) NULL+(bytes))

struct vertex {
	float x, y, z;
	float s, t;
};

// quad
const float powiekszenie = 10;
static vertex position[] = {
	{powiekszenie*(-1),powiekszenie*1,0,	0,1}, // 1 x,y,z,s,t (polozenie + wsp. teksturowania)
	{powiekszenie*(-1),powiekszenie*(-1),0,	0,0}, // 2
	{powiekszenie*1,powiekszenie*(-1),0,	1,0}, // 3

	{powiekszenie*(-1),powiekszenie*1,0,	0,1}, // 1
	{powiekszenie*1,powiekszenie*(-1),0,	1,0}, // 3
	{powiekszenie*1,powiekszenie*1,0,		1,1}  // 4
};

GLuint loadTexture(const char * filename, unsigned int width, unsigned int height)
{
	std::fstream file(filename, std::fstream::in | std::fstream::binary);
	if ( file.is_open() )
	{
		const unsigned int count = width * height * 3;
		char * data = new char[count]; // RGB

		file.read(data, count);

		GLuint texture;
		glGenTextures(1,&texture);
		glBindTexture(GL_TEXTURE_2D,texture);
		glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
		glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
		glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
		
		glTexImage2D(GL_TEXTURE_2D,0,GL_RGB,width,height,0,GL_RGB,GL_UNSIGNED_BYTE,data);

		delete [] data;

		return texture;
	}
	else
	{
		std::cout << "Nie mozna odnalezc pliku: " << filename << std::endl;
	}

	return -1;
}

void releaseTexture(GLuint texture) {
	glDeleteTextures(1, &texture);
}



/*
 *	Kontruktor: tutaj inicjalizujemy format pixeli i kontekst renderingu.
 *	Warto take pokaza okno :). To nie jest miejsce na inicjalizacje funkcji gl*.
 *	Oraz caa reszta co musi by w konstruktorze.
 */
MyGame::MyGame()
{
	TRACE();

	PixelFormat pixelFormat; // domylne parametry, podwjne buforowanie, bez antialisingu
	getGraphicsDevice().setPixelFormat(pixelFormat);

	GraphicsContext graphicsContext; // domylnie opengl 3.3 core forward
	getGraphicsDevice().setGraphicsContext(graphicsContext);

	//getGraphicsDevice().setVerticalSync(false); // wycz synchronizacje pionow

	getWindow().setDimensions(600, 600); // ustaw rozmiar okna
	getWindow().show(); // poka okno
	getWindow().update(); // odwie okno
}

/*
 *	Metoda odpalana jest raz tu po zainicjalizowaniu caej aplikacji - gotowy kontekst openGL etc.
 *	Tutaj inicjalizujemy wszystko co zwizane z OpenGL. adowanie tekstur, modeli etc.
 */
void MyGame::initialize()
{
	TRACE();
	glClearColor(0.3f, 0.5f, 0.9f, 0.0f); // ustawiamy 'kolor ta'

	glEnable(GL_DEPTH_TEST);
	//glEnable(GL_CULL_FACE);

	//glPointSize(5.0f); // rozmiar wyswietlanego punktu
	//glPolygonMode(GL_FRONT_AND_BACK, GL_LINE); // wireframe


	m_normalTexture = loadTexture("..\\Content\\normal_map.raw", 256, 256);
	ASSERT(m_normalTexture != -1);
	m_colorTexture = loadTexture("..\\Content\\colour_map.raw", 256, 256);
	ASSERT(m_colorTexture != -1);

	this->initShaders();
	this->initBuffers();
	this->initVertexPositionBuffer();

	m_camera.position = Vector3(0.0f, 2.0f, 25.0f);
	m_projection = Matrix4::createPerspective(60.0f * Math::PiDiv180, getGraphicsDevice().getViewport().getAspectRatio(), 1.0f, 10000.0f);
	m_mvpLocation = glGetUniformLocation(m_identityProgram.getHandle(), "mvp");
	m_normalTextureLoc = glGetUniformLocation(m_identityProgram.getHandle(), "normalTexture");
	m_colorTextureLoc = glGetUniformLocation(m_identityProgram.getHandle(), "colorTexture");
}

void MyGame::initShaders()
{
	VertexShader vertexShader;
	if ( !vertexShader.loadFromFile("..\\Shaders\\vertexshader.vs") )
	{
		std::printf("Vertex Shader File Error\n");
	}
	if ( !vertexShader.compile() )
	{
		vertexShader.getLog(logBuffer, 1024);
		std::printf("Vertex Shader Compile Error: %s\n", logBuffer);
	}

	/*GeometryShader geometryShader;
	if ( !geometryShader.loadFromFile("..\\Shaders\\geometryshader.gs") )
	{
		std::printf("Geometry Shader File Error\n");
	}

	if ( !geometryShader.compile() )
	{
		geometryShader.getLog(logBuffer, 1024);
		std::printf("Geometry Shader Compile Error: %s\n", logBuffer);
	}*/

	FragmentShader fragmentShader;
	if ( !fragmentShader.loadFromFile("..\\Shaders\\fragmentshader.fs") )
	{
		std::printf("Fragment Shader File Error\n");
	}

	if ( !fragmentShader.compile() )
	{
		fragmentShader.getLog(logBuffer, 1024);
		std::printf("Fragment Shader Compile Error: %s\n", logBuffer);
	}

	if ( !m_identityProgram.create() )
	{
		std::printf("Cannot create a shader program.\n");
	}

	m_identityProgram.attach(vertexShader);
	//m_identityProgram.attach(geometryShader);
	m_identityProgram.attach(fragmentShader);

	if ( !m_identityProgram.link() )
	{
		m_identityProgram.getLog(logBuffer, 1024);
		std::printf("Shader Program Linker Error: %s\n", logBuffer);
	}

	if ( !m_identityProgram.validate() )
	{
		std::printf("Shader Program Validate Error.\n");
	}
}

void MyGame::initBuffers()
{
	// wygeneruj obiekty buforw
	glGenBuffers(buffersCount, m_buffers);

	// podczep bufor, zaalokuj pami, skopiuj dane, odepnij bufor
	glBindBuffer(GL_ARRAY_BUFFER, m_buffers[0]);
		glBufferData(GL_ARRAY_BUFFER, sizeof(position), position, GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void MyGame::initVertexPositionBuffer()
{
	glGenVertexArrays(1, &m_vertexPositionBuffer);

	glBindVertexArray(m_vertexPositionBuffer);
		glBindBuffer(GL_ARRAY_BUFFER, m_buffers[0]);

		m_positionLocation = glGetAttribLocation(m_identityProgram.getHandle(), "position");
		ASSERT(m_positionLocation != -1);
		m_texcoordLocation = glGetAttribLocation(m_identityProgram.getHandle(), "texcoord");
		ASSERT(m_texcoordLocation != -1);

		glEnableVertexAttribArray(m_positionLocation);
		glEnableVertexAttribArray(m_texcoordLocation);

		glVertexAttribPointer(m_positionLocation, 3, GL_FLOAT, GL_FALSE, sizeof(vertex), BUFFER_OFFSET(NULL));
		glVertexAttribPointer(m_texcoordLocation, 2, GL_FLOAT, GL_FALSE, sizeof(vertex), BUFFER_OFFSET(3*sizeof(float)));

		glBindBuffer(GL_ARRAY_BUFFER, 0);
	glBindVertexArray(0);
}

/*
 *	Metoda odpalana jest raz tu przed zamkniciem aplikacji.
 *	Tutaj zwalniamy wszystko co zainicjalizowalimy w MyGame::initialize().
 */
void MyGame::release()
{
	releaseTexture(m_colorTexture);
	releaseTexture(m_normalTexture);

	glDeleteVertexArrays(1, &m_vertexPositionBuffer);
	glDeleteBuffers(buffersCount, m_buffers);

	TRACE();
}

/*
 *	Metoda jest odpalana za kadym razem gdy zmieni si rozmiar okna.
 *	Kady komunikat WM_SIZE od Windows spowoduj wywoanie tej metody.
 *	Dobre miejsce na np: glViewport, macierz projekcji (viewport.getAspectRatio() moe si przyda)
 */
void MyGame::reshape( const Viewport& viewport )
{
	glViewport(0, 0, viewport.getWidth(), viewport.getHeight());
	m_projection = Matrix4::createPerspective(60.0f * Math::PiDiv180, viewport.getAspectRatio(), 1.0f, 100.0f);
}

/*
 *	Metoda odpalana jest raz na kad klatk.
 *	Bardzo dobre miejsce do sprawdzania stanu myszki, klawiatury etc.
 *	Take mog by wszystkie inne rzeczy, ktre nie pasuj do update i render, a wymagaj
 *	odwierzania co klatk.
 *	elapsedTime - czas ktry upyn od ostatniej klatki, liczony w sekundach
 */
void MyGame::input( float elapsedTime )
{
	if ( getKeyboard().isKeyDown(Keys::W) ) {
		m_camera.walk(1.0f, elapsedTime);
	}

	if ( getKeyboard().isKeyDown(Keys::S) ) {
		m_camera.walk(-1.0f, elapsedTime);
	}

	if ( getKeyboard().isKeyDown(Keys::A) ) {
		m_camera.strafe(-1.0f, elapsedTime);
	}

	if ( getKeyboard().isKeyDown(Keys::D) ) {
		m_camera.strafe(1.0f, elapsedTime);
	}

	if ( getKeyboard().isKeyDown(Keys::R) ) {
		m_camera.fly(1.0f, elapsedTime);
	}

	if ( getKeyboard().isKeyDown(Keys::F) ) {
		m_camera.fly(-1.0f, elapsedTime);
	}

	if ( getKeyboard().isKeyDown(Keys::Q) ) {
		m_camera.yaw(1.0f);
	}

	if ( getKeyboard().isKeyDown(Keys::E) ) {
		m_camera.yaw(-1.0f);
	}

	if ( getKeyboard().isKeyDown(Keys::Up) ) {
		m_camera.pitch(1.0f);
	}

	if ( getKeyboard().isKeyDown(Keys::Down) ) {
		m_camera.pitch(-1.0f);
	}

	if ( getKeyboard().isKeyDown(Keys::Left) ) {
		m_camera.yaw(-1.0f);
	}

	if ( getKeyboard().isKeyDown(Keys::Right) ) {
		m_camera.yaw(1.0f);
	}

	if ( getMouse().isButtonDown(MouseButtons::Right) )
	{
		m_camera.yaw( getMouse().getRelativePosX() );
		m_camera.pitch( getMouse().getRelativePosY() );
	}

	m_view = m_camera.getViewMatrix();
	m_model = Matrix4::Identity;
	m_mvp = m_projection * m_view * m_model;
}

/*
 *	Metoda odpalana jest od jednego do kilku razy na klatk - tzw. symulacja staokrokowa.
 *	Tutaj jedynie fizyka aktorw - np. PhysX, Bullet etc.
 *	elapsedTime - stay czas, domylnie 1/60 sekundy
 */
void MyGame::update( float elapsedTime )
{
}

/*
 *	Metoda odpalana jest raz na kad klatk.
 *	Tutaj jedynie renderowanie aktorw i innych obiektw.
 *	elapsedTime - czas ktry upyn od ostatniej klatki, liczony w sekundach
 */
void MyGame::render( float elapsedTime )
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glUseProgram( m_identityProgram.getHandle() );
	glUniformMatrix4fv(m_mvpLocation, 1, GL_FALSE, m_mvp.begin());
	

	//bump mapping
	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);
	glUniform1i(m_colorTextureLoc,0);
	glBindTexture(GL_TEXTURE_2D,m_colorTexture);

	glActiveTexture(GL_TEXTURE1);
	glEnable(GL_TEXTURE_2D);
	glUniform1i(m_normalTextureLoc,1);
	glBindTexture(GL_TEXTURE_2D,m_normalTexture);

	glBindVertexArray(m_vertexPositionBuffer);
	glDrawArrays(GL_TRIANGLES,0,2*3);
	glBindVertexArray(0);

	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_2D,0);
	glDisable(GL_TEXTURE_2D); //??

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D,0);
	glDisable(GL_TEXTURE_2D); //??

	///

	glUseProgram( 0 );
	glFlush();
}