#include "matrix.h"

// *** exception objects ***
MatNonInvErr matnoninverr;
MatUniNotBound matuninotbound;
MatStaOv matstaov;
MATGLUNIMATPTR Matrix4rf::UniformMatrix3fv = 0;
MATGLUNIMATPTR Matrix4rf::UniformMatrix4fv = 0;
// *** helper functions ***
inline void Swapf(float *x, float *y) { float tmp; tmp=*x; *x=*y; *y=tmp; }
// *** low level linear algebra ***
void TransMat4x4(float *M) {
    int i, j;
    for (i=0; i<4; ++i) {
        for (j=i+1; j<4; ++j) Swapf(M+i*4+j, M+j*4+i);
    }
}
float DetMat4x4(float* M) {
    float X[12];
    X[0]=M[0]*M[5]-M[1]*M[4]; X[1]=M[0]*M[6]-M[2]*M[4];
    X[2]=M[0]*M[7]-M[3]*M[4]; X[3]=M[1]*M[6]-M[2]*M[5];
    X[4]=M[1]*M[7]-M[3]*M[5]; X[5]=M[2]*M[7]-M[3]*M[6];
    X[6]=M[8]*M[13]-M[9]*M[12]; X[7]=M[8]*M[14]-M[10]*M[12];
    X[8]=M[8]*M[15]-M[11]*M[12]; X[9]=M[9]*M[14]-M[10]*M[13];
    X[10]=M[9]*M[15]-M[11]*M[13]; X[11]=M[10]*M[15]-M[11]*M[14];
    return X[0]*X[11]-X[1]*X[10]+X[2]*X[9]+X[3]*X[8]-X[4]*X[7]+X[5]*X[6];
}
float InvMat4x4(float* M) {
    float X[12];
    float TMP[16];
    float det, deti;
    X[0]=M[0]*M[5]-M[1]*M[4]; X[1]=M[0]*M[6]-M[2]*M[4];
    X[2]=M[0]*M[7]-M[3]*M[4]; X[3]=M[1]*M[6]-M[2]*M[5];
    X[4]=M[1]*M[7]-M[3]*M[5]; X[5]=M[2]*M[7]-M[3]*M[6];
    X[6]=M[8]*M[13]-M[9]*M[12]; X[7]=M[8]*M[14]-M[10]*M[12];
    X[8]=M[8]*M[15]-M[11]*M[12]; X[9]=M[9]*M[14]-M[10]*M[13];
    X[10]=M[9]*M[15]-M[11]*M[13]; X[11]=M[10]*M[15]-M[11]*M[14];
    det = X[0]*X[11]-X[1]*X[10]+X[2]*X[9]+X[3]*X[8]-X[4]*X[7]+X[5]*X[6];
    if (fabsf(det)<1e-6) return 0.0;
    deti=1.0/det;
    TMP[0]=deti*(M[5]*X[11]-M[6]*X[10]+M[7]*X[9]);
    TMP[1]=deti*(M[2]*X[10]-M[1]*X[11]-M[3]*X[9]);
    TMP[2]=deti*(M[13]*X[5]-M[14]*X[4]+M[15]*X[3]);
    TMP[3]=deti*(M[10]*X[4]-M[9]*X[5]-M[11]*X[3]);
    TMP[4]=deti*(M[6]*X[8]-M[4]*X[11]-M[7]*X[7]);
    TMP[5]=deti*(M[0]*X[11]-M[2]*X[8]+M[3]*X[7]);
    TMP[6]=deti*(M[14]*X[2]-M[12]*X[5]-M[15]*X[1]);
    TMP[7]=deti*(M[8]*X[5]-M[10]*X[2]+M[11]*X[1]);
    TMP[8]=deti*(M[4]*X[10]-M[5]*X[8]+M[7]*X[6]);
    TMP[9]=deti*(M[1]*X[8]-M[0]*X[10]-M[3]*X[6]);
    TMP[10]=deti*(M[12]*X[4]-M[13]*X[2]+M[15]*X[0]);
    TMP[11]=deti*(M[9]*X[2]-M[8]*X[4]-M[11]*X[0]);
    TMP[12]=deti*(M[5]*X[7]-M[4]*X[9]-M[6]*X[6]);
    TMP[13]=deti*(M[0]*X[9]-M[1]*X[7]+M[2]*X[6]);
    TMP[14]=deti*(M[13]*X[1]-M[12]*X[3]-M[14]*X[0]);
    TMP[15]=deti*(M[8]*X[3]-M[9]*X[1]+M[10]*X[0]);
    memcpy(M, TMP, 16*sizeof(float));
    return det;
}
void MultMat4x4(float *A, float *B, float *C) {
    int i, j, k;
    float S;
    for (i=0; i<4; ++i) for (j=0; j<4; ++j) { 
        S=0.0; for (k=0; k<4; ++k) S+=A[i*4+k]*B[k*4+j]; C[i*4+j]=S;  
    }
}
float InvTranMat3x3From4x4(float* IN, float* OUT) {
    float det, deti;
    float X[3];
    X[0]=IN[5]*IN[10]-IN[6]*IN[9]; 
    X[1]=IN[6]*IN[8]-IN[4]*IN[10];
    X[2]=IN[4]*IN[9]-IN[5]*IN[8];
    det=IN[0]*X[0]+IN[1]*X[1]+IN[2]*X[2];
    if (fabsf(det)<1e-6) return 0.0;
    deti=1.0/det;
    OUT[0]=deti*X[0]; OUT[1]=deti*X[1]; OUT[2]=deti*X[2];
    OUT[3]=deti*(IN[2]*IN[9]-IN[1]*IN[10]);
    OUT[4]=deti*(IN[0]*IN[10]-IN[2]*IN[8]);
    OUT[5]=deti*(IN[8]*IN[1]-IN[0]*IN[9]);
    OUT[6]=deti*(IN[1]*IN[6]-IN[2]*IN[5]);
    OUT[7]=deti*(IN[2]*IN[4]-IN[0]*IN[6]);
    OUT[8]=deti*(IN[0]*IN[5]-IN[1]*IN[4]);
    return det;
}
// *** Matrix4rf ***
void Matrix4rf::Init() { uni_ena = nor_uni_ena = false; }
Matrix4rf::Matrix4rf() { Init(); }
Matrix4rf::Matrix4rf(float* c) { Init(); memcpy(coeff, c, 64); }
Matrix4rf::Matrix4rf(float M11, float M12, float M13, float M14, 
             float M21, float M22, float M23, float M24,
             float M31, float M32, float M33, float M34,
             float M41, float M42, float M43, float M44) {
        Init();
        coeff[0][0] = M11; coeff[0][1] = M12; coeff[0][2] = M13; coeff[0][3] = M14;
        coeff[1][0] = M21; coeff[1][1] = M22; coeff[1][2] = M23; coeff[1][3] = M24;
        coeff[2][0] = M31; coeff[2][1] = M32; coeff[2][2] = M33; coeff[2][3] = M34;
        coeff[3][0] = M41; coeff[3][1] = M42; coeff[3][2] = M43; coeff[3][3] = M44;
    }
Matrix4rf::Matrix4rf(const Matrix4rf& M) {
    Init(); memcpy(coeff, M.coeff, 64);
}
Matrix4rf& Matrix4rf::operator=(const Matrix4rf& M) {
    Init(); UniformMatrix3fv=UniformMatrix4fv=0; memcpy(coeff, M.coeff, 64);
}
std::string Matrix4rf::ToString() {
    std::stringstream ss(std::stringstream::in | std::stringstream::out);
    ss.precision(3);
    ss.flags(std::ios::fixed);
    for (int j=0; j<4; ++j) {
        for (int i=0; i<4; ++i) { ss << coeff[j][i] << " "; } 
        ss << "\n";
    }
    return ss.str();
}
void Matrix4rf::SetToIdentity() {
    for (int i=0; i<4; ++i) 
        for (int j=0; j<4; ++j) coeff[i][j]=float(i==j);
}
void Matrix4rf::Transpose() {
    TransMat4x4((float*)coeff);
}
Matrix4rf Matrix4rf::Transposed() {
    Matrix4rf M(*this);
    TransMat4x4((float*)M.coeff);
    return M;
}
float Matrix4rf::Invert() {
    float det;
    det = InvMat4x4((float*)coeff);
    if(det==0.0) throw matnoninverr;
    return det;
}
Matrix4rf Matrix4rf::Inverted() {
    Matrix4rf M(*this);
    if (InvMat4x4((float*)M.coeff)==0.0) throw matnoninverr;
    return M;
}
float Matrix4rf::Det() {
    return DetMat4x4((float*)coeff);
}
void Matrix4rf::Multiply(Matrix4rf& R) {
    Matrix4rf M(*this);
    MultMat4x4((float*)M.coeff, (float*)R.coeff, (float*)this->coeff);
}
void Matrix4rf::PreMultiply(Matrix4rf& L) {
    Matrix4rf M(*this);
    MultMat4x4((float*)L.coeff, (float*)M.coeff, (float*)this->coeff);
}
void Matrix4rf::SetToTranslation(float x, float y, float z) {
    SetToIdentity();
    coeff[0][3]=x; coeff[1][3]=y; coeff[2][3]=z; 
}
void Matrix4rf::AppendTranslation(float x, float y, float z) {
    for (int i=0; i<4; ++i) { 
        coeff[i][3]+=coeff[i][0]*x+coeff[i][1]*y+coeff[i][2]*z;
    }
}
void Matrix4rf::PrependTranslation(float x, float y, float z) {
    for (int i=0; i<4; ++i) { 
        coeff[0][i]+=coeff[3][i]*x;
        coeff[1][i]+=coeff[3][i]*y;
        coeff[2][i]+=coeff[3][i]*z;
    }
}
void Matrix4rf::SetToScale(float x, float y, float z) {
    SetToIdentity();
    coeff[0][0]=x; coeff[1][1]=y; coeff[2][2]=z;
}
void Matrix4rf::AppendScale(float x, float y, float z) {
    for (int i=0; i<4; ++i) { 
        coeff[0][i]*=x;
        coeff[1][i]*=y;
        coeff[2][i]*=z;
    }
}
void Matrix4rf::PrependScale(float x, float y, float z) {
    for (int i=0; i<4; ++i) { 
        coeff[i][0]*=x;
        coeff[i][1]*=y;
        coeff[i][2]*=z;
    }
}
void Matrix4rf::SetToRotation(float angle, float x, float y, float z) {
    float s, c, r, xx, yy, zz, c_, xs, ys, zs; angle*=DEG2RAD; 
    xx=x*x; yy=y*y; zz=z*z; r = 1.0/sqrt(xx+yy+zz); x*=r; y*=r; z*=r;
    s = sin(angle); c = cos(angle); c_ = 1.0-c; xs=x*s; ys=y*s; zs=z*s;
    coeff[0][0] = xx*c_+c; coeff[1][1] = yy*c_+c; coeff[2][2] = zz*c_+c;
    coeff[0][1] = x*y*c_ - zs; coeff[0][2] = x*z*c_ + ys;
    coeff[1][0] = y*x*c_ + zs; coeff[1][2] = y*z*c_ - xs;
    coeff[2][0] = x*z*c_ - ys; coeff[2][1] = y*z*c_ + xs;
    coeff[0][3]=coeff[1][3]=coeff[2][3]=0.0;
    coeff[3][0]=coeff[3][1]=coeff[3][2]=0.0; coeff[3][3]=1.0;
}
void Matrix4rf::AppendRotation(float angle, float x, float y, float z) {
    Matrix4rf M;
    M.SetToRotation(angle, x, y, z);
    this->Multiply(M);
}
void Matrix4rf::PrependRotation(float angle, float x, float y, float z) {
    Matrix4rf M;
    M.SetToRotation(angle, x, y, z);
    this->PreMultiply(M);
}
void Matrix4rf::SetToOrtho(float left, float right, float bottom, float top, float near, float far) {
    float x, y, z;
    x = 1.0/(right - left); y = 1.0/(top - bottom); z = 1.0/(far - near);
    coeff[0][0] = 2*x; coeff[0][1] = 0.0; coeff[0][2] = 0.0; coeff[0][3] = -(right+left)*x;
    coeff[1][0] = 0.0; coeff[1][1] = 2*y; coeff[1][2] = 0.0; coeff[1][3] = -(top+bottom)*y;
    coeff[2][0] = 0.0; coeff[2][1] = 0.0; coeff[2][2] = 2*z; coeff[2][3] = -(near+far)*z;
    coeff[3][0] = 0.0; coeff[3][1] = 0.0; coeff[3][2] = 0.0; coeff[3][3] = 0.0;
}
void Matrix4rf::AppendOrtho(float left, float right, float bottom, float top, float near, float far) {
    Matrix4rf M;
    M.SetToOrtho(left, right, bottom, top, near, far);
    this->Multiply(M);
}
void Matrix4rf::PrependOrtho(float left, float right, float bottom, float top, float near, float far) {
    Matrix4rf M;
    M.SetToOrtho(left, right, bottom, top, near, far);
    this->PreMultiply(M);
}
void Matrix4rf::SetToFrustum(float left, float right, float bottom, float top, float near, float far) {
    float x, y, z, n2;
    x = 1.0/(right - left); y = 1.0/(top - bottom); z = 1.0/(far - near); n2 = 2*near;
    coeff[0][0] = n2*x; coeff[0][1] = 0.0;  coeff[0][2] = (right+left)*x; coeff[0][3] = 0.0;
    coeff[1][0] = 0.0;  coeff[1][1] = n2*y; coeff[1][2] = (top+bottom)*y; coeff[1][3] = 0.0;
    coeff[2][0] = 0.0;  coeff[2][1] = 0.0;  coeff[2][2] = -(far+near)*z;  coeff[2][3] = -n2*far*z;
    coeff[3][0] = 0.0;  coeff[3][1] = 0.0;  coeff[3][2] = -1.0;           coeff[3][3] = 0.0;
}
void Matrix4rf::AppendFrustum(float left, float right, float bottom, float top, float near, float far) {
    Matrix4rf M;
    M.SetToFrustum(left, right, bottom, top, near, far);
    this->Multiply(M);
}
void Matrix4rf::PrependFrustum(float left, float right, float bottom, float top, float near, float far) {
    Matrix4rf M;
    M.SetToFrustum(left, right, bottom, top, near, far);
    this->PreMultiply(M);
}
void Matrix4rf::SetToPerspective(float fovy, float aspect, float near, float far) {
    float f, nf; f = 1.0 / tan(fovy*DEG2RAD/2.0); nf = 1.0/(near - far);
    coeff[0][0] = f/aspect; coeff[0][1] = 0.0; coeff[0][2] = 0.0;           coeff[0][3] = 0.0;
    coeff[1][0] = 0.0;      coeff[1][1] = f;   coeff[1][2] = 0.0;           coeff[1][3] = 0.0;
    coeff[2][0] = 0.0;      coeff[2][1] = 0.0; coeff[2][2] = (far+near)*nf; coeff[2][3] = 2*near*far*nf;
    coeff[3][0] = 0.0;      coeff[3][1] = 0.0; coeff[3][2] = -1.0;          coeff[3][3] = 0.0;
}
void Matrix4rf::AppendPerspective(float fovy, float aspect, float near, float far) {
    Matrix4rf M;
    M.SetToPerspective(fovy, aspect, near, far);
    this->Multiply(M);
}
void Matrix4rf::PrependPerspective(float fovy, float aspect, float near, float far) {
    Matrix4rf M;
    M.SetToPerspective(fovy, aspect, near, far);
    this->PreMultiply(M);
}
void Matrix4rf::BindUniform(int location){
    uni_ena=true;
    uni_loc=location;
}
void Matrix4rf::BindNorUniform(int location){
    nor_uni_ena=true;
    nor_uni_loc=location;
}
/*
void Matrix4rf::BindNorUniform(const char* name, unsigned prog){
    BindNorUniform(GetUniformLocation(prog, name));
}
void Matrix4rf::BindUniform(const char* name, unsigned prog){
    BindUniform(GetUniformLocation(prog, name));
}
*/
void Matrix4rf::UpdateUniforms() {
    float N[9];
    if (uni_ena && UniformMatrix4fv) {
        UniformMatrix4fv(uni_loc, 1, GL_TRUE, (float*) coeff);
        /*
        std::cout << "Sending data:";
        for (int i=0; i<4; ++i) { 
            std::cout << "\n";
            for (int j=0; j<4; ++j) { std::cout << coeff[i][j] << " "; }
        }
        std::cout << "\n";
        */
    }
    if (nor_uni_ena && UniformMatrix3fv) {
        if (InvTranMat3x3From4x4((float*)coeff, (float*)N)!=0.0) {
            UniformMatrix3fv(nor_uni_loc, 1, GL_TRUE, (float*) N);
        }
    }
}
void Matrix4rf::SetFunPtrs(MATGLUNIMATPTR UM3, MATGLUNIMATPTR UM4) {
    Matrix4rf::UniformMatrix3fv = UM3;
    Matrix4rf::UniformMatrix4fv = UM4;
}
// *** Matrix4rfStack ***
Matrix4rfStack::Matrix4rfStack(unsigned size) : Matrix4rf() {
    stack = (float*) malloc(64*size); this->size = size; pos = 0;
}
Matrix4rfStack::Matrix4rfStack(unsigned size, const Matrix4rf& M) : Matrix4rf(M) {
    stack = (float*) malloc(64*size); this->size = size; pos = 0;
}
Matrix4rfStack::Matrix4rfStack(const Matrix4rfStack& MS) : Matrix4rf((float*)MS.coeff) {
    size = MS.size; stack = (float*) malloc(64*size); pos = 0;
    memcpy(stack, MS.stack, 64*size);
}
Matrix4rfStack& Matrix4rfStack::operator=(const Matrix4rfStack& MS) {
    Resize(MS.size); memcpy(stack, MS.stack, 64*size);
    memcpy(coeff, MS.coeff, 64); uni_ena = nor_uni_ena = false; pos = MS.pos;
    Init();
}
Matrix4rfStack::~Matrix4rfStack() {
    free(stack); stack = 0;
}
void Matrix4rfStack::Pop() {
    if (pos<1) throw matstaov;
    --pos;
    memcpy(coeff, stack+16*pos, 64);
};
void Matrix4rfStack::Push() {
    if (pos+1>size) throw matstaov;
    memcpy(stack+16*pos, coeff, 64);
    ++pos;
};
void Matrix4rfStack::Resize(unsigned size) {
    this->size = size; stack = (float*) realloc(stack, 64*size);
}
