// This code contains NVIDIA Confidential Information and is disclosed to you 
// under a form of NVIDIA software license agreement provided separately to you.
//
// Notice
// NVIDIA Corporation and its licensors retain all intellectual property and
// proprietary rights in and to this software and related documentation and 
// any modifications thereto. Any use, reproduction, disclosure, or 
// distribution of this software and related documentation without an express 
// license agreement from NVIDIA Corporation is strictly prohibited.
// 
// ALL NVIDIA DESIGN SPECIFICATIONS, CODE ARE PROVIDED "AS IS.". NVIDIA MAKES
// NO WARRANTIES, EXPRESSED, IMPLIED, STATUTORY, OR OTHERWISE WITH RESPECT TO
// THE MATERIALS, AND EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES OF NONINFRINGEMENT,
// MERCHANTABILITY, AND FITNESS FOR A PARTICULAR PURPOSE.
//
// Information and code furnished is believed to be accurate and reliable.
// However, NVIDIA Corporation assumes no responsibility for the consequences of use of such
// information or for any infringement of patents or other rights of third parties that may
// result from its use. No license is granted by implication or otherwise under any patent
// or patent rights of NVIDIA Corporation. Details are subject to change without notice.
// This code supersedes and replaces all information previously supplied.
// NVIDIA Corporation products are not authorized for use as critical
// components in life support devices or systems without express written approval of
// NVIDIA Corporation.
//
// Copyright (c) 2008-2012 NVIDIA Corporation. All rights reserved.
// Copyright (c) 2004-2008 AGEIA Technologies, Inc. All rights reserved.
// Copyright (c) 2001-2004 NovodeX AG. All rights reserved.  


#ifndef PX_PHYSICS_CCT_EXTENDED
#define PX_PHYSICS_CCT_EXTENDED
/** \addtogroup character
  @{
*/

// This needs to be included in Foundation just for the debug renderer

#include "PxPhysX.h"
#include "foundation/PxTransform.h"

#ifndef PX_DOXYGEN
namespace physx
{
#endif

// This has to be done here since it also changes the top-level "Nx" API (as well as "Nv" and "Np" ones)
#define PX_BIG_WORLDS

#ifdef PX_BIG_WORLDS
typedef	double	PxExtended;
#define	PX_MAX_EXTENDED	PX_MAX_F64
#define PxExtendedAbs(x)	fabs(x)

struct PxExtendedVec3
{
	PX_INLINE	PxExtendedVec3()																	{}
	PX_INLINE	PxExtendedVec3(PxExtended _x, PxExtended _y, PxExtended _z) : x(_x), y(_y), z(_z)	{}

	PX_INLINE	bool isZero()	const
	{
		if(x!=0.0 || y!=0.0 || z!=0.0)	return false;
		return true;
	}

	PX_INLINE PxExtended	dot(const PxVec3& v) const
	{
		return x * v.x + y * v.y + z * v.z;
	}

	PX_INLINE	PxExtended distanceSquared(const PxExtendedVec3& v) const
	{
		PxExtended dx = x - v.x;
		PxExtended dy = y - v.y;
		PxExtended dz = z - v.z;
		return dx * dx + dy * dy + dz * dz;
	}

	PX_INLINE PxExtended magnitudeSquared() const
	{
		return x * x + y * y + z * z;
	}

	PX_INLINE PxExtended magnitude() const
	{
		return PxSqrt(x * x + y * y + z * z);
	}

	PX_INLINE	PxExtended	normalize()
	{
		PxExtended m = magnitude();
		if (m)
		{
			const PxExtended il =  PxExtended(1.0) / m;
			x *= il;
			y *= il;
			z *= il;
		}
		return m;
	}

	PX_INLINE	bool isFinite()	const
	{
		return PxIsFinite(x) && PxIsFinite(y) && PxIsFinite(z);
	}

	PX_INLINE	void maximum(const PxExtendedVec3& v)
	{
		if (x < v.x) x = v.x;
		if (y < v.y) y = v.y;
		if (z < v.z) z = v.z;
	}


	PX_INLINE	void minimum(const PxExtendedVec3& v)
	{
		if (x > v.x) x = v.x;
		if (y > v.y) y = v.y;
		if (z > v.z) z = v.z;
	}

	PX_INLINE	void	set(PxExtended x, PxExtended y, PxExtended z)
	{
		this->x = x;
		this->y = y;
		this->z = z;
	}

	PX_INLINE void	setPlusInfinity()
	{
		x = y = z = PX_MAX_EXTENDED;
	}

	PX_INLINE void	setMinusInfinity()
	{
		x = y = z = -PX_MAX_EXTENDED;
	}

	PX_INLINE void	cross(const PxExtendedVec3& left, const PxVec3& right)
	{
		// temps needed in case left or right is this.
		PxExtended a = (left.y * right.z) - (left.z * right.y);
		PxExtended b = (left.z * right.x) - (left.x * right.z);
		PxExtended c = (left.x * right.y) - (left.y * right.x);

		x = a;
		y = b;
		z = c;
	}

	PX_INLINE void	cross(const PxExtendedVec3& left, const PxExtendedVec3& right)
	{
		// temps needed in case left or right is this.
		PxExtended a = (left.y * right.z) - (left.z * right.y);
		PxExtended b = (left.z * right.x) - (left.x * right.z);
		PxExtended c = (left.x * right.y) - (left.y * right.x);

		x = a;
		y = b;
		z = c;
	}

	PX_INLINE PxExtendedVec3 cross(const PxExtendedVec3& v) const
	{
		PxExtendedVec3 temp;
		temp.cross(*this,v);
		return temp;
	}

	PX_INLINE void	cross(const PxVec3& left, const PxExtendedVec3& right)
	{
		// temps needed in case left or right is this.
		PxExtended a = (left.y * right.z) - (left.z * right.y);
		PxExtended b = (left.z * right.x) - (left.x * right.z);
		PxExtended c = (left.x * right.y) - (left.y * right.x);

		x = a;
		y = b;
		z = c;
	}

	PX_INLINE	PxExtendedVec3		operator-()		const
	{
		return PxExtendedVec3(-x, -y, -z);
	}

	PX_INLINE	PxExtendedVec3&		operator+=(const PxExtendedVec3& v)
	{
		x += v.x;
		y += v.y;
		z += v.z;
		return *this;
	}

	PX_INLINE	PxExtendedVec3&		operator-=(const PxExtendedVec3& v)
	{
		x -= v.x;
		y -= v.y;
		z -= v.z;
		return *this;
	}

	PX_INLINE	PxExtendedVec3&		operator+=(const PxVec3& v)
	{
		x += PxExtended(v.x);
		y += PxExtended(v.y);
		z += PxExtended(v.z);
		return *this;
	}

	PX_INLINE	PxExtendedVec3&		operator-=(const PxVec3& v)
	{
		x -= PxExtended(v.x);
		y -= PxExtended(v.y);
		z -= PxExtended(v.z);
		return *this;
	}

	PX_INLINE	PxExtendedVec3&		operator*=(const PxReal& s)
	{
		x *= PxExtended(s);
		y *= PxExtended(s);
		z *= PxExtended(s);
		return *this;
	}

	PX_INLINE	PxExtendedVec3		operator+(const PxExtendedVec3& v)	const
	{
		return PxExtendedVec3(x + v.x, y + v.y, z + v.z);
	}

	PX_INLINE	PxVec3			operator-(const PxExtendedVec3& v)	const
	{
		return PxVec3(PxReal(x - v.x), PxReal(y - v.y), PxReal(z - v.z));
	}

	PX_INLINE	PxExtended&			operator[](int index)
	{
		PX_ASSERT(index>=0 && index<=2);
		return (&x)[index];
	}


	PX_INLINE	PxExtended			operator[](int index) const
	{
		PX_ASSERT(index>=0 && index<=2);
		return (&x)[index];
	}

	PxExtended x,y,z;
};

	PX_FORCE_INLINE PxVec3 toVec3(const PxExtendedVec3& v)
	{
		return PxVec3(float(v.x), float(v.y), float(v.z));
	}

#else
// Big worlds not defined

typedef	PxVec3		PxExtendedVec3;
typedef	PxReal		PxExtended;
#define	PX_MAX_EXTENDED	PX_MAX_F32
#define PxExtendedAbs(x)	fabsf(x)
#endif

#ifndef PX_DOXYGEN
} // namespace physx
#endif

/** @} */
#endif
