using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;

using System.IO;

namespace JacekMatulewski.Util
{
	/// <summary>
	/// Komponent prezentujcy zawarto wybranego katalogu oraz umoliwiajcy eksplorowanie dyskw komputera.
	/// </summary>
	/// <remarks>
	/// Komponent <b>FileListBox</b> uzupenia luk w oryginalnych bibliotekach platformy .NET, w ktrej nie ma komponentw tego typu. Komponent jest w znacznym stopniu konfigurowalny.
	/// </remarks>
	public class FileListBox : System.Windows.Forms.UserControl
	{
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;
		private System.Windows.Forms.ListBox listBox1;

		#region Pola prywatne
		//wewnetrzne
		private string[] listaKatalogow=null;
		private string[] listaPlikow=null;
		private string[] listaDyskow=null;
		private bool pokazujDwieKropki=true;
		private FileSystemWatcher wartownik=null;
		//konfigurowanie komponentu
		private string sciezkaKatalogu=null;
		private bool uwzglednijKatalogi=true;
		private bool uwzglednijPliki=true;
		private bool uwzglednijDyski=true;
		private bool uwzglednijKatalogNadrzedny=true;
		private string filtr=null;
		private bool mozliweZmienianieKatalogow=true;
		#endregion

		#region Metody prywatne
		private void PobierzZawartoscKatalogu()
		{
			if (sciezkaKatalogu==null)
				sciezkaKatalogu=Directory.GetCurrentDirectory();

			if (!Directory.Exists(sciezkaKatalogu))
				throw new Exception("Katalog "+sciezkaKatalogu+" nie istnieje!");

			pokazujDwieKropki=(sciezkaKatalogu!=Path.GetPathRoot(sciezkaKatalogu) && uwzglednijKatalogNadrzedny);

			listBox1.Items.Clear();

			if (uwzglednijKatalogi)
				{
				if (pokazujDwieKropki) listBox1.Items.Add("[..]");
				listaKatalogow=Directory.GetDirectories(sciezkaKatalogu);
				Array.Sort(listaKatalogow);
				//listBox1.Items.AddRange(listaKatalogow);
				//for (int i=0;i<listaKatalogow.Length;i++)
					//listBox1.Items.Add("["+Path.GetFileName(listaKatalogow[i])+"]");
				foreach (string katalog in listaKatalogow)
					listBox1.Items.Add("["+Path.GetFileName(katalog)+"]");
				}
			if (uwzglednijPliki)
				{
				listaPlikow=Directory.GetFiles(sciezkaKatalogu);
				if (filtr!=null) listaPlikow=Directory.GetFiles(sciezkaKatalogu,filtr);
				Array.Sort(listaPlikow);
				//listBox1.Items.AddRange(listaPlikow);
				//for (int i=0;i<listaPlikow.Length;i++)
					//listBox1.Items.Add(Path.GetFileName(listaPlikow[i]));
				foreach (string plik in listaPlikow)
					listBox1.Items.Add(Path.GetFileName(plik));
				}
			if (uwzglednijDyski)
				{
				listaDyskow=Directory.GetLogicalDrives();
				//listBox1.Items.AddRange(listaDyskow);
				foreach (string dysk in listaDyskow)
					listBox1.Items.Add("<"+dysk.Substring(0,2)+">");

				}
		}
		#endregion


    	/// <summary>
		/// Tworzy now instancj klasy <b>FileListBox</b>.
		/// </summary>
		public FileListBox()
		{
			// This call is required by the Windows.Forms Designer.
			InitializeComponent();
			//
			PobierzZawartoscKatalogu();
			UstawWartownika();
		}

		/// <summary>
		/// Czyci wszystkie zasoby uywane przez obiekt.
		/// </summary>
		protected override void Dispose(bool disposing)
		{
			if (disposing)
			{
				if (components != null)
					components.Dispose();
			}
			base.Dispose(disposing);
		}

		#region Component Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.listBox1 = new System.Windows.Forms.ListBox();
			this.SuspendLayout();
			// 
			// listBox1
			// 
			this.listBox1.Dock = System.Windows.Forms.DockStyle.Fill;
			this.listBox1.ItemHeight = 16;
			this.listBox1.Location = new System.Drawing.Point(0, 0);
			this.listBox1.Name = "listBox1";
			this.listBox1.Size = new System.Drawing.Size(150, 148);
			this.listBox1.TabIndex = 0;
			this.listBox1.KeyDown += new System.Windows.Forms.KeyEventHandler(this.listBox1_KeyDown);
			this.listBox1.MouseDown += new System.Windows.Forms.MouseEventHandler(this.listBox1_MouseDown);
			this.listBox1.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.listBox1_KeyPress);
			this.listBox1.DoubleClick += new System.EventHandler(this.listBox1_DoubleClick);
			this.listBox1.MouseHover += new System.EventHandler(this.listBox1_MouseHover);
			this.listBox1.KeyUp += new System.Windows.Forms.KeyEventHandler(this.listBox1_KeyUp);
			this.listBox1.MouseUp += new System.Windows.Forms.MouseEventHandler(this.listBox1_MouseUp);
			this.listBox1.MouseMove += new System.Windows.Forms.MouseEventHandler(this.listBox1_MouseMove);
			this.listBox1.MouseEnter += new System.EventHandler(this.listBox1_MouseEnter);
			this.listBox1.MouseLeave += new System.EventHandler(this.listBox1_MouseLeave);
			this.listBox1.SelectedIndexChanged += new System.EventHandler(this.listBox1_SelectedIndexChanged);
			// 
			// FileListBox
			//
			this.Controls.Add(this.listBox1);
			this.Name = "FileListBox";
			this.ResumeLayout(false);
		}
		#endregion

		private void listBox1_DoubleClick(object sender, System.EventArgs e)
		{
			//przekazanie zdarzenia DoubleClick
			this.OnDoubleClick(e);

			//wywolanie zdarzenia FileDoubleClick jezeli klikniety jest plik
			int przesuniecie=(pokazujDwieKropki && uwzglednijKatalogi)?1:0;
			int numer=listBox1.SelectedIndex-przesuniecie; //+1 dla [..]
			int poczatekPlikow=(uwzglednijKatalogi?listaKatalogow.Length:0);
			int poczatekDyskow=poczatekPlikow+(uwzglednijPliki?listaPlikow.Length:0);

			//zmiana katalogu
			string nowaSciezkaKatalogu=null;
			if (numer==-1) nowaSciezkaKatalogu=sciezkaKatalogu+"\\..";
			if (numer>=0 && numer<poczatekPlikow) nowaSciezkaKatalogu=listaKatalogow[numer];
			if (numer>=poczatekDyskow) nowaSciezkaKatalogu=listaDyskow[numer-poczatekDyskow];
			if (nowaSciezkaKatalogu!=null && Directory.Exists(nowaSciezkaKatalogu) && mozliweZmienianieKatalogow)
				{
				sciezkaKatalogu=Path.GetFullPath(nowaSciezkaKatalogu);
				PobierzZawartoscKatalogu();
				this.OnDirectoryPathChanged(this,e);
				wartownik.Path=sciezkaKatalogu;
				}

			//podwojne klikniecie pliku
			string sciezkaPliku=null;
			if (numer>=poczatekPlikow && numer<poczatekDyskow) sciezkaPliku=listaPlikow[numer-poczatekPlikow];
			if (sciezkaPliku!=null && File.Exists(sciezkaPliku))
				this.OnFileDoubleClicked(this,e,Path.GetFullPath(sciezkaPliku));
		}

		#region Wlasnosci
		/// <summary>
		/// Wasno okrelajca ciek do katalogu prezentowanego w komponencie.
		/// </summary>
		/// <value>
		/// acuch zawierajcy pen ciek do katalogu widocznego w komponencie.
		/// </value>
		[
			Category("Directory"),
			Description("The DirectoryPath determines relative or absolute path of directory visible at component.")
		]
		public string DirectoryPath
		{
			set
			{
				this.sciezkaKatalogu = value;
				PobierzZawartoscKatalogu();
				this.OnDirectoryPathChanged(this,new System.EventArgs());
				wartownik.Path=sciezkaKatalogu;
			}
			get
			{
				return this.sciezkaKatalogu;
			}
		}

		/// <summary>
		/// Filtr stosowany do plikw widocznych w komponencie.
		/// </summary>
		/// <value>
		/// acuch zawierajcy mask filtru.
		/// </value>
		[
			Category("Directory"),
			Description("The Filter determines the mask of files in the list.")
		]
		public string Filter
		{
			set
			{
				this.filtr = value;
				PobierzZawartoscKatalogu();
			}
			get
			{
				return this.filtr;
			}
		}


		/// <summary>
		/// Wasno okrelajca, czy symbol katalogu nadrzdnego (dwie kropki) jest widoczny (tylko jeeli <b>DirectoriesVisible</b> ma warto <b>true</b>.
		/// </summary>
		/// <value>
		/// Jeeli <b>DoubleDotVisible</b> oraz <b>DirectoriesVisible</b> maj wartoci <b>true</b> symbol katalogu nadrzdnego jest widoczny.
		/// </value>
		[
			Category("Directory"),
			Description("The DoubleDotVisible determines if symbol of upper directory should be displayed.")
		]
		public bool DoubleDotVisible
		{
			set
			{
				this.uwzglednijKatalogNadrzedny = value;
				PobierzZawartoscKatalogu();
			}
			get
			{
				return this.uwzglednijKatalogNadrzedny;
			}
		}

		/// <summary>
		/// Wasno okrelajca, czy wywietlana jest lista katalogw.
		/// </summary>
		/// <value>
		/// Jeeli <b>DirectoriesVisible</b> ma warto <b>true</b> komponent prezentuje list podkatalogw katalogu wskazanego przez wasno <b>DirectoryPath</b>.
		/// </value>
		[
			Category("Directory"),
			Description("The DirectoriesVisible determines if list of directories should be displayed.")
		]
		public bool DirectoriesVisible
		{
			set
			{
				this.uwzglednijKatalogi = value;
				PobierzZawartoscKatalogu();
			}
			get
			{
				return this.uwzglednijKatalogi;
			}
		}

		/// <summary>
		/// Wasno okrelajca, czy wywietlana jest lista plikw.
		/// </summary>
		/// <value>
		/// Jeeli <b>FilesVisible</b> ma warto <b>true</b> komponent prezentuje list plikw znajdujcych si w katalogu wskazanyn przez wasno <b>DirectoryPath</b>.
		/// </value>
		[
			Category("Directory"),
			Description("The FilesVisible determines if list of files should be displayed.")
		]
		public bool FilesVisible
		{
			set
			{
				this.uwzglednijPliki = value;
				PobierzZawartoscKatalogu();
			}
			get
			{
				return this.uwzglednijPliki;
			}
		}

		/// <summary>
		/// Wasno okrelajca, czy wywietlana jest lista wszystkich dyskw dostpnych w komputerze.
		/// </summary>
		/// <value>
		/// Jeeli <b>DrivesVisible</b> ma warto <b>true</b> komponent prezentuje list dyskw dostpnych w komputerze.
		/// </value>
		[
			Category("Directory"),
			Description("The DrivesVisible determines if list of drives should be displayed.")
		]
		public bool DrivesVisible
		{
			set
			{
				this.uwzglednijDyski = value;
				PobierzZawartoscKatalogu();
			}
			get
			{
				return this.uwzglednijDyski;
			}
		}

		/// <summary>
		/// Wasno okrelajca, czy po dwukrotnym klikniciu katalogu widocznego w komponencie pokazana zostanie jego zawarto, a wic czy moliwa jest eksploracja dyskw komputera.
		/// </summary>
		/// <value>
		/// Jeeli <b>DirectoryChangeAllowed</b> ma warto <b>true</b> komponent pozwala na interaktywn zmian katalogu okrelonego przez wasno <b>DirectoryPath</b>. W przeciwnym razie jest to moliwe jedynie przez jawn zmian wasnoci <b>DirectoryPath</b>.
		/// </value>
		[
			Category("Directory"),
			Description("The DirectoryChangeAllowed determines if user can change current directory by double-clicking on directory in the list.")
		]
		public bool DirectoryChangeAllowed
		{
			set
			{
				mozliweZmienianieKatalogow = value;
			}
			get
			{
				return mozliweZmienianieKatalogow;
			}
		}

		/// <summary>
		/// Wasno pozwalajca na odczytanie absolutnej cieki do zaznaczonego lub ostatnio zaznaczonego pliku.
		/// </summary>
		/// <value>
		/// acuch zawierajcy pen ciek do zaznaczonego w komponencie pliku.
		/// </value>
		[
			Category("Directory"),
			Description("The FileName expose the full path and filename to selected item.")
		]
		public string FileName
		{
			get
			{
				int przesuniecie=(pokazujDwieKropki && uwzglednijKatalogi)?1:0;
				int numer=listBox1.SelectedIndex-przesuniecie; //+1 dla [..]
				int poczatekPlikow=(uwzglednijKatalogi?listaKatalogow.Length:0);
				int poczatekDyskow=poczatekPlikow+(uwzglednijPliki?listaPlikow.Length:0);

				string pelnaSciezka=null; //ta wartosc zostanie gdy brak zaznaczonego elementu
				if (numer==-1 && przesuniecie==1) pelnaSciezka=sciezkaKatalogu+"\\..";
				if (numer>=0 && numer<poczatekPlikow) pelnaSciezka=listaKatalogow[numer];
				if (numer>=poczatekPlikow && numer<poczatekDyskow) pelnaSciezka=listaPlikow[numer-poczatekPlikow];
				if (numer>=poczatekDyskow) pelnaSciezka=listaDyskow[numer-poczatekDyskow];

				if (pelnaSciezka==null) return "";

				return Path.GetFullPath(pelnaSciezka);
			}
		}

		/// <summary>
		/// Wasno okrelajca, czy "wartownik" prezentowanego katalogu jest wczony. "Wartownik" umoliwia ledzenie wszystkich zmian w katalogu, ktre powoduj zmiany w licie plikw i w wyniku tego aktualizacj listy widocznej w komponencie.
		/// </summary>
		/// <value>
		/// Jeeli <b>DirectoryWatchingEnabled</b> ma warto <b>true</b> komponent bdzie sam aktualizowa swoj zawarto w przypadku zmian w katalogu. W przeciwnym przypadku uytkownik moe tego dokonac sam wywoujc metod <see cref="Refresh">Refresh</see>.
		/// </value>
		[
			Category("Directory"),
			Description("The DirectoryWatchingEnabled determines if files and subdirectories list is automatically uptadated.")
		]
		public bool DirectoryWatchingEnabled
		{
			set
			{
				wartownik.EnableRaisingEvents = value;
				if (wartownik.EnableRaisingEvents) PobierzZawartoscKatalogu();

			}
			get
			{
				return wartownik.EnableRaisingEvents;
			}
		}
		#endregion

		#region Zdarzenia
		private void listBox1_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			this.OnClick(e);
			this.OnSelectedFileChanged(sender,e);
		}

		//Zdarzenie SelectedFileChanged
		//1. deklaracja typu metody zdarzeniowej
		public delegate void SelectedFileChangedEventHandler(object sender,System.EventArgs e); //deklaracja typu metody zdarzeniowej
		//2. deklaracja zdarzenia i opis do niej w atrybucie
		/// <summary>
		/// Zgaszane przy zmianie zaznaczonego pliku.
		/// </summary>
		/// <seealso cref="FileName"></seealso>
		/// <seealso cref="OnSelectedFileChanged"></seealso>
		/// <seealso cref="SelectedFileChangedEventHandler"></seealso>
		[
			Category("Directory"),
			Description("Occures when file selectecion is changed.")
		]
		public event SelectedFileChangedEventHandler SelectedFileChanged;
		//3. pomocnicza metoda wywolujaca (launch method)
		protected virtual void OnSelectedFileChanged(object sender,System.EventArgs e)
		{
			//koniecznie trzeba sprawdzic, czy ze zdarzeniem jest w ogole zwiazana jakas metoda
			if (SelectedFileChanged!=null) SelectedFileChanged(this,e);
		}

		//Zdarzenie DirectoryPathChanged
		/// <summary>
		/// Zgaszane przy zmianie prezentowanego katalogu.
		/// </summary>
		/// <seealso cref="DirectoryPath"></seealso>
		/// <seealso cref="OnDirectoryPathChanged"></seealso>
		/// <seealso cref="SelectedFileChangedEventHandler"></seealso>
		[
			Category("Directory"),
			Description("Occures when directory presented in component is changed.")
		]
		public event SelectedFileChangedEventHandler DirectoryPathChanged;
		protected virtual void OnDirectoryPathChanged(object sender,System.EventArgs e)
		{
			if (DirectoryPathChanged!=null) DirectoryPathChanged(this,e);
		}

		//Zdarzenie FileDoubleClicked
		public delegate void FileDoubleClickedEventHandler(object sender,System.EventArgs e,string fileName);
		/// <summary>
		/// Zgaszane po dwukrotnym klikniciu pliku.
		/// </summary>
		/// <seealso cref="OnFileDoubleClicked"></seealso>
		/// <seealso cref="FileDoubleClickedEventHandler"></seealso>
		[
			Category("Directory"),
			Description("Occures when one of the files in the list is double clicked.")
		]
		public event FileDoubleClickedEventHandler FileDoubleClicked;
		protected virtual void OnFileDoubleClicked(object sender,System.EventArgs e,string fileName)
		{
			if (FileDoubleClicked!=null) FileDoubleClicked(this,e,fileName);
		}
		#endregion

		//nadpisanie metody Refresh
		/// <summary>
		/// Metoda odwieajca komponent wraz z zawartoci prezentowanego katalogu.
		/// </summary>
		public override void Refresh()
		{
			base.Refresh(); //czy to konieczne?
			PobierzZawartoscKatalogu();
		}


		//sledzenie zmian w prezentowanym katalogu
		private void UstawWartownika()
		{
			//nic nie robimy jezeli obiekt juz istnieje
			if (wartownik!=null) return;

			wartownik = new FileSystemWatcher();

			wartownik.Path = sciezkaKatalogu;

			//sledzone czynnosci
			wartownik.NotifyFilter = NotifyFilters.FileName | NotifyFilters.DirectoryName;

			//wiazanie zdarzen wartownika z metoda Refresh
			wartownik.Changed += new FileSystemEventHandler(PrzyZmianie);
			wartownik.Created += new FileSystemEventHandler(PrzyZmianie);
			wartownik.Deleted += new FileSystemEventHandler(PrzyZmianie);
			wartownik.Renamed += new RenamedEventHandler(PrzyZmianie_Renamed);

			//aktywacja obserwowania
			wartownik.EnableRaisingEvents = true;
		}

		//wywolujace metody pomocnicza
		private void PrzyZmianie(object source, FileSystemEventArgs e)
		{
			this.PobierzZawartoscKatalogu();
		}
		private void PrzyZmianie_Renamed(object source, RenamedEventArgs e)
		{
			this.PobierzZawartoscKatalogu();
		}

		//listBox1_MouseDown, listBox1_MouseUp i listBox1_MouseMove sa w bloku drag'n'drop

		private void listBox1_MouseEnter(object sender, System.EventArgs e)
		{
			this.OnMouseEnter(e);
		}

		private void listBox1_MouseHover(object sender, System.EventArgs e)
		{
			this.OnMouseHover(e);
		}

		private void listBox1_MouseLeave(object sender, System.EventArgs e)
		{
			this.OnMouseLeave(e);
		}

		private void listBox1_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			this.OnKeyDown(e);
		}

		private void listBox1_KeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e)
		{
			this.OnKeyPress(e);
		}

		private void listBox1_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			this.OnKeyUp(e);
		}

		/// <summary>
		/// Okrela, czy zaznaczany moe by aden, jeden lub wiele plikw jednoczenie.
		/// </summary>
		[
			Category("Behavior"),
			Description("Indicates if the list box is to be single-select, multi-select, or unselectable.")
		]
		public SelectionMode SelectionMode
		{
			set
			{
				listBox1.SelectionMode = value;
			}
			get
			{
				return listBox1.SelectionMode;
			}
		}


		#region drag'n'drop
		//wlasnosc StartDragAutomatically
		private bool startDragAutomatically=false;
		/// <summary>
		/// Wasno okrelajca, czy proces Drag'n'drop ma zosta rozpoczty automatycznie po przeniesieniu elementu na odlego okrelon przez wasno <see cref="DragDistance">DragDistance</see>.
		/// </summary>
		[
			Category("Drag Drop"),
			Description("Determines if the drag'n'drop process begin automatically after dragging element for 5 pixels.")
		]
		public bool StartDragAutomatically
		{
			set
			{
				startDragAutomatically = value;
			}
			get
			{
				return startDragAutomatically;
			}
		}

		//wlasnosc DragDistance
		private int dragDistance=5;
		/// <summary>
		/// Wasno okrelajca ilo pikseli, po ktrych ruch myszy z przycinitym lewym klawiszem jest traktowany jako przenoszenie elementu.
		/// </summary>
		/// <seealso cref="MouseDrag"></seealso>
		/// <seealso cref="StartDrag"></seealso>
		[
			Category("Drag Drop"),
			Description("Determines the distance the item must be dragged with mouse to begin drag'n'drop process (if StartDragAutomatically is set to true).")
		]
		public int DragDistance
		{
			set
			{
				dragDistance = value;
			}
			get
			{
				return dragDistance;
			}
		}

		//metody i pola zwiazane z obsluga automatycznego uruchomienia drag'n'drop po przesunieciu o 5 pixeli
		private bool mouseClicked=false;
		private Point mouseClickedPos;

		private void listBox1_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			//przekazanie zdarzenia z listBox1 do FileListBox
			this.OnMouseDown(e);

			//zwiazane z drag'n'drop
			if (e.Button==MouseButtons.Left && System.IO.File.Exists(this.FileName))
				{
				mouseClicked=true;
				mouseClickedPos=new Point(e.X,e.Y);
				}
		}

		private void listBox1_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			//przekazanie zdarzenia z listBox1 do FileListBox
			this.OnMouseUp(e);

			//zwiazane z drag'n'drop
			mouseClicked=false;
		}

		private void listBox1_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			//przekazanie zdarzenia z listBox1 do FileListBox
			this.OnMouseMove(e);

			//zwiazane z drag'n'drop
			if (mouseClicked)
				{
				int dx=e.X-mouseClickedPos.X;
				int dy=e.Y-mouseClickedPos.Y;
				double distance=Math.Sqrt(dx*dx+dy*dy);
				if (distance>=dragDistance)
					{
					this.OnMouseDrag(sender,e);
					if (startDragAutomatically)
						{
						this.OnStartDrag(sender,e,this.FileName);
						this.DoDragDrop(this.FileName,DragDropEffects.Copy);
						}
					mouseClicked=false;
					}
				}
		}

		//definicja zdarzenia MouseDrag
		/// <summary>
		/// Zgaszane w momencie przesunicia elementu na odlego okrelon przez wasno <see cref="DragDistance">DragDistance</see>.
		/// </summary>
		[
			Category("Mouse"),
			Description("Occures when the mouse is moved number of pixels determined by DragDistance with left button pressed.")
		]
		public event MouseEventHandler MouseDrag;
		protected virtual void OnMouseDrag(object sender,System.Windows.Forms.MouseEventArgs e)
		{
			if (MouseDrag!=null) MouseDrag(this,e);
		}

		//definicja zdarzenia StartDrag
		/// <summary>
		/// Zgaszane w momencie rozpoczcia procesu Drag'n'drop.
		/// </summary>
		[
			Category("Drag Drop"),
			Description("Occures when user has begun to drag the file in the list.")
		]
		public event FileDoubleClickedEventHandler StartDrag;
		protected virtual void OnStartDrag(object sender,System.EventArgs e,string fileName)
		{
			if (StartDrag!=null) StartDrag(this,e,fileName);
		}
		#endregion
	}
}
